import { Box, Typography } from '@mui/material';
import Divider from '@mui/material/Divider';
import Grid from '@mui/material/Grid';
import dayjs from 'dayjs';
import { useEffect } from 'react';
import { useFormContext } from 'react-hook-form';
import { Field } from 'src/components/hook-form';
import { SETAHUN, SETAHUN_TEXT } from 'src/sections/bupot-21-26/constant';

type PPHDipotongProps = {
  isPengganti?: boolean;
  kodeObjectPajak: {
    value: string;
    label: string;
  }[];
  fgFasilitasOptions: {
    value: string;
    label: string;
  }[];
};

const setahunOptions = [
  { value: SETAHUN.SETAHUN, label: SETAHUN_TEXT[SETAHUN.SETAHUN] },
  { value: SETAHUN.DISETAHUNKAN, label: SETAHUN_TEXT[SETAHUN.DISETAHUNKAN] },
  { value: SETAHUN.BAGIAN_TAHUN, label: SETAHUN_TEXT[SETAHUN.BAGIAN_TAHUN] },
];

const RincianPenghasilan = ({
  kodeObjectPajak,
  fgFasilitasOptions,
  isPengganti,
}: PPHDipotongProps) => {
  const { watch, setValue } = useFormContext();
  const isMetodePemotonganSeTahun = watch('metodePemotongan') === '1';
  const tanggalPemotongan = watch('tglPemotongan');
  const masaPajakAwal = watch('masaPajakAwal');
  const masaPajakAkhir = watch('masaPajakAkhir');

  useEffect(() => {
    if (!isPengganti) {
      if (tanggalPemotongan) {
        const date = dayjs(tanggalPemotongan);
        setValue('tahunPajak', date.format('YYYY'));
        setValue('masaPajak', date.format('MM'));
      } else {
        setValue('tahunPajak', '');
        setValue('masaPajak', '');
      }
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [tanggalPemotongan, !isPengganti]);

  const fgFasilitas = watch('fgFasilitas');

  return (
    <Grid container rowSpacing={2} columnSpacing={2}>
      {/* Divider */}
      <Grid size={{ md: 12 }}>
        <Divider sx={{ fontWeight: 'bold' }} textAlign="left">
          Rincian Penghasilan dan Penghitungan PPh Pasal 21
        </Divider>
      </Grid>
      <Grid size={{ md: !isMetodePemotonganSeTahun ? 6 : 12 }}>
        <Field.RadioGroup
          row
          name="metodePemotongan"
          label="Jenis Pemotongan"
          options={setahunOptions}
          slotProps={{
            radio: {
              slotProps: {
                input: {
                  readOnly: isPengganti,
                },
              },
            },
          }}
        />
      </Grid>
      {!isMetodePemotonganSeTahun && (
        <Grid size={{ md: 6 }}>
          <Field.Text name="noBupotSebelumnya" label="Nomor Bupot Sebelumnya" />
        </Grid>
      )}
      <Grid size={{ md: 3 }}>
        <Field.DatePicker
          name="tglPemotongan"
          label="Tanggal Pemotongan"
          format="DD/MM/YYYY"
          maxDate={dayjs()}
        />
      </Grid>
      <Grid size={{ md: 3 }}>
        <Field.DatePicker
          name="tahunPajak"
          label="Tahun Pajak"
          view="year"
          format="YYYY"
          minDate={dayjs('2025')}
          readOnly={isPengganti}
          openTo="year"
        />
      </Grid>
      <Grid size={{ md: 2.5 }}>
        <Field.DatePicker
          name="masaPajakAwal"
          label="Masa Pajak Awal"
          view="month"
          format="MM"
          openTo="month"
          maxDate={dayjs(masaPajakAkhir)}
          readOnly={isMetodePemotonganSeTahun || isPengganti}
        />
      </Grid>
      <Grid size={{ md: 1 }}>
        <Box className="flex items-center" sx={{ justifyContent: 'center', alignItems: 'center' }}>
          <Typography variant="body1" textAlign="center" fontWeight="bold">
            s.d.
          </Typography>
        </Box>
      </Grid>
      <Grid size={{ md: 2.5 }}>
        <Field.DatePicker
          name="masaPajakAkhir"
          label="Masa Pajak Akhir"
          view="month"
          format="MM"
          openTo="month"
          minDate={dayjs(masaPajakAwal)}
          readOnly={isMetodePemotonganSeTahun || isPengganti}
        />
      </Grid>

      {/* Kode objek pajak */}
      <Grid size={{ md: 6 }}>
        <Field.Autocomplete name="kdObjPjk" label="Kode Objek Pajak" options={kodeObjectPajak} />
      </Grid>

      {/* Fasilitas */}
      <Grid size={{ md: 6 }}>
        <Field.Autocomplete name="fgFasilitas" label="Fasilitas" options={fgFasilitasOptions} />
      </Grid>

      {/* Dokumen lainnya */}
      <Grid size={{ md: 6 }}>
        <Field.Text
          name="noDokLainnya"
          label="Nomor Dokumen Lainnya"
          disabled={['9', ''].includes(fgFasilitas.value)}
          sx={{ '& .MuiInputBase-root.Mui-disabled': { backgroundColor: '#f6f6f6' } }}
        />
      </Grid>
    </Grid>
  );
};

export default RincianPenghasilan;
