import Button from '@mui/material/Button';
import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
import { Link as RouterLink } from 'react-router-dom';
import { paths } from 'src/routes/paths';
import React, { useCallback, useEffect, useMemo, useRef, useState, startTransition } from 'react';
import { useNavigate } from 'react-router';
import type {
  GridColDef,
  GridFilterModel,
  GridRowSelectionModel,
  GridSortModel,
  GridToolbarProps,
  GridPaginationModel,
} from '@mui/x-data-grid-premium';
import { DataGridPremium, useGridApiRef } from '@mui/x-data-grid-premium';
import { unstable_batchedUpdates } from 'react-dom';

import {
  ArticleTwoTone,
  AutorenewTwoTone,
  DeleteSweepTwoTone,
  EditNoteTwoTone,
  FileOpenTwoTone,
  HighlightOffTwoTone,
  UploadFileTwoTone,
  SwapHorizontalCircleTwoTone,
} from '@mui/icons-material';
import { enqueueSnackbar } from 'notistack';
import { usePaginationStore } from '../store/paginationStore';
import StatusChip from '../components/StatusChip';
import { useDebounce, useThrottle } from 'src/shared/hooks/useDebounceThrottle';
import useAdvancedFilter from '../hooks/useAdvancedFilterFakturPK';
import { CustomToolbar } from '../components/CustomToolbar';
import useGetDokumenLainKeluaran, {
  formatDateToDDMMYYYY,
} from '../hooks/useGetDokumenLainKeluaran';
import { formatRupiah } from 'src/shared/FormatRupiah/FormatRupiah';
import {
  DETAIL_TRANSAKSI_TEXT,
  DOKUMEN_TRANSAKSI_TEXT,
  KD_TRANSAKSI_TEXT,
  KODE_DOKUMEN_TEXT,
} from '../constant';
import ModalCetakDokumenLainKeluaran from '../components/dialog/ModalCetakDokumenLainKeluaran';
import ModalCancelDokumenLainKeluaran from '../components/dialog/ModalCancelDokumenLainKeluaran';
import ModalUploadDokumenLainKeluaran from '../components/dialog/ModalUploadDokumenLainKeluaran';
import ModalDeleteDokumenLainKeluaran from '../components/dialog/ModalDeleteDokumenLainKeluaran';

export type IColumnGrid = GridColDef & {
  field:
    | 'statusdokumen'
    | 'nomordokumen'
    | 'kdtransaksi'
    | 'detailtransaksi'
    | 'dokumentransaksi'
    | 'kodedokumen'
    | 'nomordokumendiganti'
    | 'npwppembeli'
    | 'namapembeli'
    | 'tanggaldokumen'
    | 'masapajak'
    | 'tahunpajak'
    | 'jumlahdpp'
    | 'jumlahppn'
    | 'jumlahppnbm'
    | 'tanggalapproval'
    | 'keterangantambahan'
    | 'namapenandatangan'
    | 'referensi'
    | 'created_by'
    | 'created_at'
    | 'updated_by'
    | 'updated_at'
    | 'alamatpembeli'
    | 'npwppenjual';
  // valueOptions?: string[];
  valueOptions?: readonly (string | number | { value: string | number; label: string })[];
};

export function DlkListView() {
  const apiRef = useGridApiRef();
  const navigate = useNavigate();

  const tableKey = 'faktur-pk';

  const { tables, filters, setPagination, resetPagination, setFilter } =
    usePaginationStore.getState();

  const [filterModel, setFilterModel] = useState<GridFilterModel>({
    items: filters[tableKey]?.items ?? [],
  });

  const [sortModel, setSortModel] = useState<GridSortModel>([]);
  const [rowSelectionModel, setRowSelectionModel] = useState<GridRowSelectionModel | undefined>(
    undefined
  );

  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState<boolean>(false);
  const [isUploadModalOpen, setIsUploadModalOpen] = useState<boolean>(false);
  const [isCancelModalOpen, setIsCancelModalOpen] = useState<boolean>(false);
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  const [isReturOpen, setIsReturOpen] = useState(false);
  const [isPreviewOpen, setIsPreviewOpen] = useState<boolean>(false);
  const [previewPayload, setPreviewPayload] = useState<Record<string, any> | undefined>(undefined);

  const dataSelectedRef = useRef<any[]>([]);
  const [selectionVersion, setSelectionVersion] = useState(0);
  const [localPagination, setLocalPagination] = useState({
    page: tables[tableKey]?.page ?? 0,
    pageSize: tables[tableKey]?.pageSize ?? 10,
  });

  const { buildAdvancedFilter, buildRequestParams } = useAdvancedFilter();

  const page = tables[tableKey]?.page ?? 0;
  const pageSize = tables[tableKey]?.pageSize ?? 10;

  const isSyncingRef = useRef(false);
  const isEqual = (a: any, b: any) => a === b || JSON.stringify(a) === JSON.stringify(b);

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const newStoreItems = state.filters[tableKey]?.items ?? [];
      const localItems = filterModel.items ?? [];

      if (!isEqual(newStoreItems, localItems)) {
        isSyncingRef.current = true;
        setFilterModel({ items: newStoreItems });
        queueMicrotask(() => (isSyncingRef.current = false));
      }
    });

    return () => unsub();
  }, [filterModel.items]);

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const storePage = state.tables[tableKey]?.page ?? 0;
      const storePageSize = state.tables[tableKey]?.pageSize ?? 10;

      setLocalPagination((prev) =>
        prev.page !== storePage || prev.pageSize !== storePageSize
          ? { page: storePage, pageSize: storePageSize }
          : prev
      );
    });
    return () => unsub();
  }, []);

  useEffect(() => {
    if (isSyncingRef.current) return;
    const currentStore = usePaginationStore.getState().filters[tableKey]?.items ?? [];
    if (!isEqual(currentStore, filterModel.items)) {
      setFilter(tableKey, { items: filterModel.items });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [filterModel]);

  const params = useMemo(() => {
    const advanced = buildAdvancedFilter(filterModel.items);

    const baseParams = {
      page,
      limit: pageSize,
      sortingMode: sortModel[0]?.field ?? '',
      sortingMethod: sortModel[0]?.sort ?? '',
    };

    return buildRequestParams(baseParams, advanced);
  }, [page, pageSize, sortModel, filterModel.items, buildAdvancedFilter, buildRequestParams]);

  const { data, isFetching, refetch } = useGetDokumenLainKeluaran({
    params,
  });
  const idStatusMapRef = useRef<Map<string | number, string>>(new Map());
  const rows = useMemo(() => (data?.data || []).filter(Boolean), [data?.data]);
  const totalRows = Number(data?.total ?? 0);

  useEffect(() => {
    try {
      const m = new Map<string | number, string>();
      (rows || []).forEach((r: any) => {
        const id = String(r.id ?? r.internal_id ?? '');
        m.set(id, r?.fgStatus ?? '');
      });
      idStatusMapRef.current = m;
    } catch {
      idStatusMapRef.current = new Map();
    }
  }, [rows]);

  const handlePaginationChange = (model: GridPaginationModel) => {
    setLocalPagination(model);
    setPagination(tableKey, {
      page: model.page,
      pageSize: model.pageSize,
    });
  };

  const debouncedFilterChange = useDebounce((model: GridFilterModel) => {
    setFilterModel(model);
  }, 400);

  const debouncedSortChange = useDebounce((model: GridSortModel) => {
    setSortModel(model);
    resetPagination(tableKey);
  }, 400);

  type Status =
    | 'DRAFT'
    | 'APPROVED'
    | 'WAITING FOR AMENDMENT'
    | 'AMENDED'
    | 'WAITING FOR CANCELLATION';
  type StatusOption = { value: Status; label: string };
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const statusOptions: StatusOption[] = [
    { value: 'DRAFT', label: 'Draft' },
    { value: 'APPROVED', label: 'Normal' },
    { value: 'WAITING FOR AMENDMENT', label: 'Normal Pengganti' },
    { value: 'AMENDED', label: 'Diganti' },
    { value: 'WAITING FOR CANCELLATION', label: 'Batal' },
  ];

  const columns = useMemo<IColumnGrid[]>(
    () => [
      {
        field: 'statusdokumen',
        headerName: 'Status',
        width: 200,
        type: 'singleSelect',
        valueOptions: statusOptions.map((opt) => opt.value),
        // valueFormatter: ({ value }: { value: string }) => {
        //   const option = statusOptions.find((opt) => opt.value === value);
        //   return option ? option.label : value;
        // },
        renderCell: ({ value, row }) => (
          <StatusChip value={value} fgpelunasan={row.fgpelunasan} fguangmuka={row.fguangmuka} />
        ),
      },
      { field: 'nomordokumen', headerName: 'Nomor Dokumen', width: 150 },
      {
        field: 'kdtransaksi',
        headerName: 'Kode Transaksi',
        width: 150,
        renderCell: ({ value }) => KD_TRANSAKSI_TEXT[value] ?? value,
      },

      {
        field: 'detailtransaksi',
        headerName: 'Detail Transaksi',
        width: 150,
        renderCell: ({ value }) => DETAIL_TRANSAKSI_TEXT[value] ?? value,
      },
      {
        field: 'dokumentransaksi',
        headerName: 'Dokumen Transaksi',
        width: 150,
        renderCell: ({ value }) => DOKUMEN_TRANSAKSI_TEXT[value] ?? value,
      },
      {
        field: 'kodedokumen',
        headerName: 'Kode Dokumen',
        width: 150,
        renderCell: ({ value }) => KODE_DOKUMEN_TEXT[value] ?? value,
      },
      {
        field: 'nomordokumendiganti',
        headerName: 'Nomor Dokumen Diganti',
        width: 150,
      },
      {
        field: 'npwppembeli',
        headerName: 'NPWP',
        width: 150,
      },
      {
        field: 'namapembeli',
        headerName: 'Nama',
        width: 200,
      },
      {
        field: 'tanggaldokumen',
        headerName: 'Tanggal Dokumen',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.tanggaldokumen),
      },
      {
        field: 'masapajak',
        headerName: 'Masa',
        width: 150,
      },
      {
        field: 'tahunpajak',
        headerName: 'Tahun',
        width: 150,
      },
      {
        field: 'jumlahdpp',
        headerName: 'DPP',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.jumlahdpp),
      },
      {
        field: 'jumlahppn',
        headerName: 'PPN',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.jumlahppn),
      },
      {
        field: 'jumlahppnbm',
        headerName: 'PPnBM',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.jumlahppnbm),
      },
      {
        field: 'tanggalapproval',
        headerName: 'Tanggal Approval',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.tanggalapproval),
      },
      { field: 'referensi', headerName: 'Referensi', width: 150 },
      { field: 'created_by', headerName: 'User Perekam', width: 150 },
      {
        field: 'created_at',
        headerName: 'Tanggal Rekam',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.created_at),
      },
      { field: 'updated_by', headerName: 'User Pengubah', width: 150 },
      {
        field: 'updated_at',
        headerName: 'Tanggal Ubah',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.updated_at),
      },
      { field: 'keterangantambahan', headerName: 'Keterangan', width: 150 },
      { field: 'namapenandatangan', headerName: 'Penandatangan', width: 150 },
      { field: 'alamatpembeli', headerName: 'Alamat Pembeli ', width: 150 },
      { field: 'npwppenjual', headerName: 'NPWP Penjual', width: 150 },
    ],
    [statusOptions]
  );

  const handleEditData = useCallback(
    (type = 'ubah') => {
      const selectedRow = dataSelectedRef.current[0];
      if (!selectedRow) return;
      navigate(`/faktur/dokumen-lain/pajak-keluaran/${selectedRow.id}/${type}`);
    },

    [navigate]
  );

  const throttledSelectionChange = useThrottle((newSelection: any) => {
    if (!apiRef.current) return;
    const ids =
      newSelection?.ids instanceof Set ? Array.from(newSelection.ids) : newSelection || [];
    const selectedData = ids.map((id: any) => apiRef.current!.getRow(id)).filter(Boolean);

    unstable_batchedUpdates(() => {
      dataSelectedRef.current = selectedData;
      setRowSelectionModel(newSelection);
      setSelectionVersion((v) => v + 1);
    });
  }, 150);

  useEffect(() => {
    const api = apiRef.current;
    if (!api) return;
    const unsubscribe = api.subscribeEvent('rowsSet', () => {
      const exec = () => {
        const ids =
          api.state?.rowSelection?.ids instanceof Set ? Array.from(api.state.rowSelection.ids) : [];
        const updatedSelected = ids.map((id) => api.getRow(id)).filter(Boolean);
        dataSelectedRef.current = updatedSelected;
        setSelectionVersion((v) => v + 1);
      };
      if ((window as any).requestIdleCallback) (window as any).requestIdleCallback(exec);
      else setTimeout(exec, 0);
    });
    // eslint-disable-next-line consistent-return
    return () => unsubscribe();
  }, [apiRef]);

  const validatedActions = useMemo(() => {
    const dataSelected = dataSelectedRef.current;
    const count = dataSelected.length;
    const hasSelection = count > 0;

    if (!hasSelection) {
      return {
        canDetail: false,
        canEdit: false,
        canDelete: false,
        canUpload: false,
        canReplacement: false,
        canCancel: false,
      };
    }

    const allDraft = dataSelected.every((d) => d.statusdokumen === 'DRAFT');
    const allApproved = dataSelected.every((d) => d.statusdokumen === 'APPROVED');

    const item = dataSelected[0];
    const isSingleReplacement = count === 1 && item.statusdokumen === 'APPROVED';

    return {
      canDetail: count === 1,
      canEdit: count === 1 && allDraft,
      canDelete: hasSelection && allDraft,
      canUpload: hasSelection && allDraft,
      canReplacement: isSingleReplacement,
      canCancel: hasSelection && allApproved,
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [selectionVersion]);

  const handleOpenPreview = () => {
    const selectedRow = dataSelectedRef.current?.[0];
    if (!selectedRow) {
      enqueueSnackbar('Pilih 1 baris untuk melihat detail', { variant: 'warning' });
      return;
    }

    console.log(selectedRow);

    setPreviewPayload(selectedRow);
    setIsPreviewOpen(true);
  };

  const actions = useMemo(
    () => [
      [
        {
          title: 'Refresh List',
          icon: <AutorenewTwoTone sx={{ width: 26, height: 26 }} />,
          func: () =>
            startTransition(() => {
              resetPagination(tableKey);
              void refetch();
            }),
        },
        {
          title: 'Edit',
          icon: <EditNoteTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => handleEditData('ubah'),
          disabled: !validatedActions.canEdit,
        },
        {
          title: 'Detail',
          icon: <ArticleTwoTone sx={{ width: 26, height: 26 }} />,
          func: handleOpenPreview,
          disabled: !validatedActions.canDetail,
        },
        {
          title: 'Hapus',
          icon: <DeleteSweepTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsDeleteModalOpen(true),
          disabled: !validatedActions.canDelete,
        },
      ],
      [
        {
          title: 'Upload',
          icon: <UploadFileTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsUploadModalOpen(true),
          disabled: !validatedActions.canUpload,
        },
        {
          title: 'Pengganti',
          icon: <FileOpenTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => handleEditData('pengganti'),
          disabled: !validatedActions.canReplacement,
        },
        {
          title: 'Batal',
          icon: <HighlightOffTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsCancelModalOpen(true),
          disabled: !validatedActions.canCancel,
        },
      ],
      [
        {
          title: 'Retur',
          icon: <SwapHorizontalCircleTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsReturOpen(true),
          // disabled: !validatedActions.canRetur,
        },
      ],
    ],
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [validatedActions, refetch, handleEditData]
  );

  const pinnedColumns = useMemo(
    () => ({ left: ['__check__', 'statusdokumen', 'nomordokumen'] }),
    []
  );

  const ToolbarWrapper: React.FC<GridToolbarProps> = useCallback(
    (gridToolbarProps) => (
      <CustomToolbar
        actions={actions}
        columns={columns}
        filterModel={filterModel}
        setFilterModel={setFilterModel}
        statusOptions={statusOptions}
        {...gridToolbarProps}
      />
    ),
    [actions, columns, filterModel, setFilterModel, statusOptions]
  );

  return (
    <>
      <DashboardContent>
        <CustomBreadcrumbs
          heading="Dokumen Lain Keluaran"
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'e-Faktur Dokumen Lain Keluaran' },
          ]}
          action={
            <Button component={RouterLink} to={paths.faktur.rekamDlk} variant="contained">
              Rekam Data
            </Button>
          }
        />

        <DataGridPremium
          apiRef={apiRef}
          checkboxSelection
          rows={rows || []}
          getRowId={(row: any) => {
            if (!row) return '';
            return String(row.id ?? row.internal_id ?? '');
          }}
          columns={columns}
          loading={isFetching}
          rowCount={totalRows}
          pagination
          paginationMode="server"
          paginationModel={localPagination}
          initialState={{
            pagination: {
              paginationModel: {
                page,
                pageSize,
              },
            },
            columns: {
              columnVisibilityModel: {
                jenisInvoice: false, // hide on load
              },
            },
          }}
          onPaginationModelChange={handlePaginationChange}
          pageSizeOptions={[5, 10, 15, 25, 50, 100]}
          filterMode="server"
          onFilterModelChange={debouncedFilterChange}
          sortingMode="server"
          onSortModelChange={debouncedSortChange}
          rowSelectionModel={rowSelectionModel}
          onRowSelectionModelChange={throttledSelectionChange}
          pinnedColumns={pinnedColumns}
          cellSelection
          showToolbar
          slots={{ toolbar: ToolbarWrapper }}
          sx={{
            border: 1,
            borderColor: 'divider',
            borderRadius: 2,
            mt: 3,
            '& .MuiDataGrid-cell': {
              borderColor: 'divider',
              userSelect: 'text',
              cursor: 'text',
            },
            '& .MuiDataGrid-columnHeaders': { borderColor: 'divider' },
          }}
          onColumnVisibilityModelChange={(newModel) => {
            if (newModel['__check__'] === false) {
              newModel['__check__'] = true;
            }
            apiRef.current?.setColumnVisibilityModel(newModel);
          }}
          disableVirtualization
        />
      </DashboardContent>

      {isDeleteModalOpen && (
        <ModalDeleteDokumenLainKeluaran
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogDelete={isDeleteModalOpen}
          setIsOpenDialogDelete={setIsDeleteModalOpen}
          successMessage="Data berhasil dihapus"
          onConfirmDelete={async () => {
            await refetch();
          }}
        />
      )}

      {isUploadModalOpen && (
        <ModalUploadDokumenLainKeluaran
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogUpload={isUploadModalOpen}
          setIsOpenDialogUpload={setIsUploadModalOpen}
          successMessage="Data berhasil diupload"
          onConfirmUpload={async () => {
            await refetch();
          }}
        />
      )}

      {isCancelModalOpen && (
        <ModalCancelDokumenLainKeluaran
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogCancel={isCancelModalOpen}
          setIsOpenDialogCancel={setIsCancelModalOpen}
          successMessage="Data berhasil dibatalkan"
          onConfirmCancel={async () => {
            await refetch();
          }}
        />
      )}

      {isPreviewOpen && (
        <ModalCetakDokumenLainKeluaran
          payload={previewPayload}
          isOpen={isPreviewOpen}
          onClose={() => {
            setIsPreviewOpen(false);
            setPreviewPayload(undefined);
          }}
        />
      )}
    </>
  );
}
