import z from 'zod';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { isValidPhoneNumber } from 'react-phone-number-input/input';

import Box from '@mui/material/Box';
import Card from '@mui/material/Card';
import Grid from '@mui/material/Grid';
import Typography from '@mui/material/Typography';

import { toast } from 'src/components/snackbar';
import { Form, Field, schemaUtils } from 'src/components/hook-form';

import { useSelector } from 'react-redux';
import type { RootState } from 'src/store';
import type { User } from 'src/auth/types/user';

// ----------------------------------------------------------------------

export type UpdateUserSchemaType = z.infer<typeof UpdateUserSchema>;

export const UpdateUserSchema = z.object({
  customer_name: z.string().min(1, { error: 'Name is required!' }),
  email: schemaUtils.email(),
  phone: schemaUtils.phoneNumber({ isValid: isValidPhoneNumber }),
  address: z.string().min(1, { error: 'Address is required!' }),
});

// ----------------------------------------------------------------------

export function AccountGeneral() {
  const user = useSelector((state: RootState) => state.user);
  console.log(user);

  const currentUser: User = {
    customer_name: user?.data.customer_name,
    email: user?.data.email,
    phone: user?.data.phone,
    address: user?.data.address,
  };

  const defaultValues: UpdateUserSchemaType = {
    customer_name: '',
    email: '',
    phone: '',
    address: '',
  };

  const methods = useForm({
    mode: 'all',
    resolver: zodResolver(UpdateUserSchema),
    defaultValues,
    values: currentUser,
  });

  const {
    handleSubmit,
    // formState: { isSubmitting },
  } = methods;

  const onSubmit = handleSubmit(async (data) => {
    try {
      await new Promise((resolve) => setTimeout(resolve, 500));
      toast.success('Update success!');
      console.info('DATA', data);
    } catch (error) {
      console.error(error);
    }
  });

  const styleCustom = {
    '& .MuiInputBase-input.Mui-disabled': {
      WebkitTextFillColor: '#000 !important',
      color: '#000 !important',
    },
    '& .MuiInputLabel-root.Mui-disabled': {
      color: '#000 !important',
    },
    '& .MuiOutlinedInput-root.Mui-disabled fieldset': {
      borderColor: '#403535ff',
    },
  };

  return (
    <Form methods={methods} onSubmit={onSubmit}>
      <Box
        sx={{
          mb: 3,
        }}
      >
        <Typography variant="h4">User Profile</Typography>
      </Box>
      <Grid container spacing={3} sx={{ flexDirection: 'column' }}>
        <Grid>
          <Card
            sx={{
              p: 2,
              background: 'linear-gradient(to right, #143b88, #1976D2)',
              borderRadius: '16px',
              mb: 2,
            }}
          >
            <Typography
              sx={{
                fontSize: '24px',
                color: '#fff',
                pb: '4px',
              }}
            >
              {user.data.customer_name}
            </Typography>
            <Typography
              sx={{
                fontSize: '16',
                color: '#BBDEFB',
              }}
            >
              {user.data.email}
            </Typography>
          </Card>
        </Grid>

        <Grid>
          <Card sx={{ p: 3 }}>
            <Box
              sx={{
                rowGap: 3,
                columnGap: 2,
                display: 'grid',
                gridTemplateColumns: { xs: 'repeat(1, 1fr)', sm: 'repeat(2, 1fr)' },
              }}
            >
              <Field.Text sx={styleCustom} name="customer_name" label="Name" disabled />
              <Field.Text sx={styleCustom} name="email" label="Email address" disabled />
              <Field.Text
                sx={styleCustom}
                name="phone"
                type="number"
                label="Phone Number"
                disabled
              />
              <Field.Text sx={styleCustom} name="address" label="Address" disabled />
            </Box>
          </Card>
        </Grid>
      </Grid>
    </Form>
  );
}
