import { z } from 'zod';

export const BarangJasaSchema = z
  .object({
    type: z.string().nonempty('Type wajib diisi'),
    kdBrgJasa: z.string().nonempty('Kode Barang Jasa wajib diisi'),
    namaBrgJasa: z.string().trim().nonempty('Nama Barang/Jasa wajib diisi'),
    satuanBrgJasa: z.string().nonempty('Satuan Barang Jasa wajib diisi'),

    jmlBrgJasa: z
      .string()
      .trim()
      .nonempty('Jumlah Barang wajib diisi')
      .refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
        message: 'Jumlah Barang wajib diisi dengan angka lebih dari 0',
      }),

    hargaSatuan: z.string().trim().nonempty('Harga Satuan wajib diisi'),
    //   .refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    //     message: 'Harga Satuan wajib diisi dengan angka lebih dari 0',
    //   }),

    totalHarga: z.string().trim().nonempty('Harga Total wajib diisi'),

    diskon: z
      .string()
      .trim()
      .nonempty('Diskon wajib diisi')
      .refine((val) => !isNaN(Number(val)) && Number(val) >= 0, {
        message: 'Diskon wajib diisi dengan angka valid',
      }),

    fgPmk: z.string().optional(),
    dpp: z.string().nonempty('DPP wajib diisi'),
    tarifPpn: z.string().nonempty('Tarif PPN wajib diisi'),

    dppLain: z
      .string()
      .min(1, 'DPP Nilai Lain wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'DPP Nilai Lain harus berupa angka' }),

    ppn: z
      .string()
      .min(1, 'PPN wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'PPN harus berupa angka' }),

    ppnbm: z
      .string()
      .min(1, 'PPnBM wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'PPnBM harus berupa angka' }),

    tarifPpnbm: z
      .string()
      .min(1, 'Tarif PPnBM wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'Tarif PPnBM harus berupa angka' }),
  })
  .refine(
    (data) => {
      const hargaTotal = Number(data.totalHarga);
      const diskon = Number(data.diskon);
      if (isNaN(hargaTotal) || isNaN(diskon)) return true;
      return diskon <= hargaTotal;
    },
    {
      message: 'Diskon tidak boleh lebih dari Harga Total',
      path: ['diskon'],
    }
  )
  .superRefine((data, ctx) => {
    const dpp = Number(data.dpp || 0);

    // Validasi setiap field agar tidak lebih besar dari DPP
    const fieldsToCheck = [
      { key: 'dppLain', label: 'DPP Nilai Lain' },
      { key: 'ppn', label: 'PPN' },
      { key: 'ppnbm', label: 'PPnBM' },
      { key: 'tarifPpnbm', label: 'Tarif PPnBM' },
    ] as const;

    fieldsToCheck.forEach(({ key, label }) => {
      const val = Number(data[key] || 0);
      if (!isNaN(val) && val > dpp) {
        ctx.addIssue({
          code: 'custom',
          path: [key],
          message: `${label} tidak boleh lebih dari jumlah DPP`,
        });
      }
    });
  });

export type TBarangJasaSchema = z.infer<typeof BarangJasaSchema>;
