import { zodResolver } from '@hookform/resolvers/zod';
import { LoadingButton } from '@mui/lab';
import Grid from '@mui/material/Grid';
import { useState, useEffect } from 'react';
import { FormProvider, useForm } from 'react-hook-form';
import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
import { paths } from 'src/routes/paths';
import HeadingRekam from 'src/shared/components/HeadingRekam';
import z from 'zod';
// import Identitas from '../components/rekamSsp/Identitas';
import Divider from '@mui/material/Divider';
import FormSkeleton from 'src/shared/skeletons/FormSkeleton';
import PphDipotong from '../components/rekamDigunggung/PphDipotong';
import useGetKodeObjekPajak from '../hooks/useGetKodeObjekPajakDigunggung';
import DokumenReferensi from '../components/rekamDigunggung/DokumenReferensi';
import Agreement from 'src/shared/components/agreement/Agreement';
import Stack from '@mui/material/Stack';
import PanduanSspRekam from '../components/rekamDigunggung/PanduanSspRekam';
import useSaveSsp from '../hooks/useSaveDigunggung';
import { enqueueSnackbar } from 'notistack';
import { useNavigate, useParams } from 'react-router';
import useUpload from '../hooks/useUpload';
import { useGetDigunggungById } from '../hooks/useGetDigunggung';
import Identitas from '../components/rekamDigunggung/Identitas';
import ModalUploadDigunggung from '../components/dialog/ModalUploadDigunggung';

const bpuDigunggungSchema = z
  .object({
    tglPemotongan: z.string().nonempty('Tanggal Pemotongan harus diisi'),
    thnPajak: z.string().nonempty('Tahun Pajak harus diisi'),
    msPajak: z.string().nonempty('Masa Pajak harus diisi'),
    kdObjPjk: z.string().nonempty('Kode Objek Pajak harus diisi'),
    fgFasilitas: z.string().nonempty('Fasilitas harus diisi'),
    noDokLainnya: z.string().optional(),
    dpp: z.string().nonempty('Jumlah Penghasilan Bruto harus diisi'),
    tarif: z.union([z.string().nonempty('Tarif harus diisi'), z.number()]),
    pphDipotong: z.string().nonempty('PPh Yang Dipotong/Dipungut harus diisi'),
    namaDok: z.string().nonempty('Nama Dokumen harus diisi'),
    nomorDok: z.string().nonempty('Nomor Dokumen harus diisi'),
    tglDok: z.string().nonempty('Tanggal Dokumen harus diisi'),
    idTku: z.string().nonempty('Cabang harus diisi'),

    pasalPph: z.string().optional(),
    statusPph: z.string().optional(),
    kap: z.union([z.string(), z.number()]).optional(),
    kjs: z.union([z.string(), z.number()]).optional(),
  })
  .superRefine((data, ctx) => {
    // Field dianggap DISABLED kalau fgFasilitas kosong ('') atau '9'
    const isDisabled = ['', '9'].includes(data.fgFasilitas);

    // Jika tidak disabled, berarti aktif → wajib isi
    if (!isDisabled && (!data.noDokLainnya || data.noDokLainnya.trim() === '')) {
      ctx.addIssue({
        path: ['noDokLainnya'],
        code: 'custom',
        message: 'No Dokumen Lainnya harus diisi',
      });
    }
  });

const DigunggungRekamView = () => {
  const { id, type } = useParams<{ id?: string; type?: 'ubah' | 'pengganti' | 'new' }>();

  const [isOpenPanduan, setIsOpenPanduan] = useState<boolean>(false);
  const [isCheckedAgreement, setIsCheckedAgreement] = useState<boolean>(false);
  const [isFormPrefilled, setIsFormPrefilled] = useState<boolean>(false);
  const [formInitialized, setFormInitialized] = useState<boolean>(false);
  const [isUploadModalOpen, setIsUploadModalOpen] = useState<boolean>(false);

  const isEdit = type === 'ubah';
  const isPengganti = type === 'pengganti';

  const { data: existingDigunggung, isLoading: isLoadingDigunggung } = useGetDigunggungById(id!, {
    enabled: !!id && (isEdit || isPengganti),
  });

  const { data: kodeObjekPajak, isLoading: isLoadingKop } = useGetKodeObjekPajak();
  const navigate = useNavigate();

  type BpuDigunggungFormData = z.infer<typeof bpuDigunggungSchema>;

  const saveSsp = useSaveSsp({
    onSuccess: () => enqueueSnackbar('Data berhasil disimpan', { variant: 'success' }),
  });

  const uploadNr = useUpload();

  const handleOpenPanduan = () => setIsOpenPanduan(!isOpenPanduan);

  const defaultValues: BpuDigunggungFormData = {
    tglPemotongan: '',
    thnPajak: '',
    msPajak: '',
    kdObjPjk: '',
    fgFasilitas: '',
    noDokLainnya: '',
    dpp: '',
    tarif: '',
    pphDipotong: '',
    namaDok: '',
    nomorDok: '',
    tglDok: '',
    idTku: '',
  };

  const methods = useForm<BpuDigunggungFormData>({
    mode: 'all',
    resolver: zodResolver(bpuDigunggungSchema),
    defaultValues,
  });

  const { reset, handleSubmit } = methods;

  useEffect(() => {
    if (isEdit || isPengganti) {
      if (existingDigunggung && !isLoadingKop) {
        // 🧩 Normalisasi nilai numeric ke string sebelum reset
        const normalized = {
          ...existingDigunggung,
          // dpp:
          //   existingDigunggung.dpp !== null && existingDigunggung.dpp !== undefined
          //     ? String(existingDigunggung.dpp)
          //     : '',
          tarif:
            existingDigunggung.tarif !== null && existingDigunggung.tarif !== undefined
              ? String(existingDigunggung.tarif)
              : '',
          pphDipotong:
            existingDigunggung.pphDipotong !== null && existingDigunggung.pphDipotong !== undefined
              ? String(existingDigunggung.pphDipotong)
              : '',
          masaPajak:
            existingDigunggung.msPajak !== null && existingDigunggung.msPajak !== undefined
              ? String(existingDigunggung.msPajak)
              : '',
          namaDok: existingDigunggung.namaDok ?? '',
          nomorDok: existingDigunggung.nomorDok ?? '',
          tglDok: existingDigunggung.tglDok ?? '',
        };

        reset(normalized);
        setIsFormPrefilled(true);
        setFormInitialized(true);
      }
    } else {
      setIsFormPrefilled(false);
      setFormInitialized(true);
    }
  }, [existingDigunggung, isLoadingKop, isEdit, isPengganti, reset]);

  const buildPayload = (values: BpuDigunggungFormData) => {
    if (isEdit) {
      return {
        ...values,
        id,
        isPengganti: false,
      };
    }

    if (isPengganti) {
      return {
        ...values,
        id,
        isPengganti: true,
        idBupot: existingDigunggung?.idBupot,
        noBupot: existingDigunggung?.noBupot,
        revNo: existingDigunggung?.revNo ?? 0,
      };
    }

    return {
      ...values,
      isPengganti: false,
    };
  };

  const onSubmit = async (values: BpuDigunggungFormData) => {
    try {
      const payload = buildPayload(values);

      await saveSsp.mutateAsync(payload);

      enqueueSnackbar(
        isEdit
          ? 'Data berhasil diperbarui'
          : isPengganti
            ? 'Data pengganti berhasil disimpan'
            : 'Data berhasil disimpan',
        { variant: 'success' }
      );

      navigate('/unifikasi/digunggung');
    } catch (error: any) {
      enqueueSnackbar(error.message || 'Gagal menyimpan data', { variant: 'error' });
      console.error('❌ saveDigunggung error:', error);
    }
  };

  const handleSaveAndUpload = async (values: BpuDigunggungFormData) => {
    try {
      const payload = buildPayload(values);
      const res: any = await saveSsp.mutateAsync(payload);

      const savedId = res?.[0]?.id;
      if (!savedId) throw new Error('ID dokumen tidak ditemukan dari hasil save');

      await uploadNr.mutateAsync({ id: savedId });

      enqueueSnackbar('Data berhasil disimpan dan diupload', { variant: 'success' });
      navigate('/unifikasi/digunggung');
    } catch (error: any) {
      enqueueSnackbar(error.message || 'Gagal save & upload data', { variant: 'error' });
      console.error('❌ Upload error:', error);
    }
  };

  if (isLoadingDigunggung || isLoadingKop || !formInitialized) {
    return (
      <DashboardContent>
        <CustomBreadcrumbs
          heading={
            isEdit
              ? 'Ubah Bupot Unifikasi Digunggung'
              : isPengganti
                ? 'Bupot Pengganti'
                : 'Tambah Bupot Digunggung'
          }
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'e-Bupot Digunggung', href: paths.unifikasi.digunggung },
            { name: isEdit ? 'Ubah' : isPengganti ? 'Pengganti' : 'Tambah' },
          ]}
        />

        <HeadingRekam
          label={
            isEdit
              ? 'Ubah Data Bukti Pemotongan/Pemungutan PPh Digunggung'
              : isPengganti
                ? 'Rekam Bukti Pengganti PPh Digunggung'
                : 'Rekam Data Bukti Potong PPh Digunggung'
          }
        />
        <FormSkeleton numberOfRows={8} />
      </DashboardContent>
    );
  }

  return (
    <DashboardContent>
      <CustomBreadcrumbs
        heading={
          isEdit
            ? 'Ubah Bupot Digunggung'
            : isPengganti
              ? 'Bupot Digunggung Pengganti'
              : 'Tambah Bupot Digunggung'
        }
        links={[
          { name: 'Dashboard', href: paths.dashboard.root },
          { name: 'e-Bupot Digunggung', href: paths.unifikasi.digunggung },
          { name: isEdit ? 'Ubah' : isPengganti ? 'Pengganti' : 'Tambah' },
        ]}
      />

      <HeadingRekam
        label={
          isEdit
            ? 'Ubah Data Bukti Pemotongan/Pemungutan PPh Digunggung'
            : isPengganti
              ? 'Rekam Bukti Pengganti PPh Digunggungi'
              : 'Rekam Data Bukti Potong PPh Digunggung'
        }
      />

      <Grid container columnSpacing={2}>
        <Grid size={{ xs: isOpenPanduan ? 8 : 11 }}>
          <FormProvider {...methods}>
            <form onSubmit={handleSubmit(onSubmit)}>
              <Identitas isPengganti={isPengganti} existingDigunggung={existingDigunggung} />
              <PphDipotong
                kodeObjectPajak={kodeObjekPajak?.data ?? []}
                isFormPrefilled={isFormPrefilled}
              />
              <DokumenReferensi />
              <Divider />
              <Grid size={12}>
                <Agreement
                  isCheckedAgreement={isCheckedAgreement}
                  setIsCheckedAgreement={setIsCheckedAgreement}
                  text="Dengan ini saya menyatakan bahwa Bukti Pemotongan/Pemungutan Unifikasi telah saya isi dengan benar secara elektronik sesuai dengan"
                />
              </Grid>
              <Stack direction="row" gap={2} justifyContent="end" marginTop={2}>
                <LoadingButton
                  type="button"
                  loading={saveSsp.isPending}
                  disabled={!isCheckedAgreement}
                  variant="outlined"
                  sx={{ color: '#143B88' }}
                  onClick={() => {
                    methods.handleSubmit(
                      async (values) => {
                        await onSubmit(values);
                      },
                      (errors) => {
                        console.error('❌ Validation errors:', errors);
                      }
                    )();
                  }}
                >
                  Save as Draft
                </LoadingButton>
                <LoadingButton
                  type="button"
                  disabled={!isCheckedAgreement}
                  //   onClick={() => methods.handleSubmit(handleSaveAndUpload)()}
                  onClick={() => setIsUploadModalOpen(true)}
                  loading={uploadNr.isPending}
                  variant="contained"
                  sx={{ background: '#143B88' }}
                >
                  Save and Upload
                </LoadingButton>
              </Stack>
            </form>
          </FormProvider>
        </Grid>
        <Grid size={{ xs: isOpenPanduan ? 4 : 1 }}>
          <PanduanSspRekam handleOpen={handleOpenPanduan} isOpen={isOpenPanduan} />
        </Grid>
      </Grid>

      {isUploadModalOpen && (
        <ModalUploadDigunggung
          isOpenDialogUpload={isUploadModalOpen}
          setIsOpenDialogUpload={setIsUploadModalOpen}
          onConfirmUpload={() => handleSaveAndUpload(methods.getValues())}
        />
      )}
    </DashboardContent>
  );
};

export default DigunggungRekamView;
