import Button from '@mui/material/Button';
import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
// import { RouterLink } from 'src/routes/components';
import { Link as RouterLink } from 'react-router-dom';
import { paths } from 'src/routes/paths';
import React, { useCallback, useEffect, useMemo, useRef, useState, startTransition } from 'react';
import { useNavigate } from 'react-router';
import type {
  GridColDef,
  GridFilterModel,
  GridRowId,
  GridRowSelectionModel,
  GridSortModel,
  GridToolbarProps,
  GridPaginationModel,
} from '@mui/x-data-grid-premium';
import { DataGridPremium, useGridApiRef } from '@mui/x-data-grid-premium';
import { unstable_batchedUpdates } from 'react-dom';

import {
  ArticleTwoTone,
  AutorenewTwoTone,
  DeleteSweepTwoTone,
  EditNoteTwoTone,
  FileOpenTwoTone,
  HighlightOffTwoTone,
  UploadFileTwoTone,
} from '@mui/icons-material';
// import { CustomToolbar } from '../components/CustomToolbar';
import { formatRupiah } from 'src/shared/FormatRupiah/FormatRupiah';
import { FG_STATUS_DN } from '../constant';
import ModalDeleteNr from '../components/dialog/ModalDeleteNr';
import ModalUploadNr from '../components/dialog/ModalUploadNr';
import ModalCancelNr from '../components/dialog/ModalCancelNr';
import ModalCetakPdfNr from '../components/dialog/ModalCetakPdfNr';
import { useGetNr } from '../hooks/useGetNr';
import { enqueueSnackbar } from 'notistack';
import { usePaginationStore } from '../store/paginationStore';
import StatusChip from '../components/StatusChip';
import { useDebounce, useThrottle } from 'src/shared/hooks/useDebounceThrottle';
import useGetKodeObjekPajak from '../hooks/useGetKodeObjekPajakNr';
import useAdvancedFilter from '../hooks/useAdvancedFilterNr';
import { CustomToolbar } from '../components/CustomToolbar';
import { useSelector } from 'react-redux';
import type { RootState } from 'src/store';

export type IColumnGrid = GridColDef & {
  field:
    | 'fgStatus'
    | 'noBupot'
    | 'masaPajak'
    | 'tahunPajak'
    | 'kodeObjekPajak'
    | 'pasalPPh'
    | 'npwpPemotong'
    | 'namaDipotong'
    | 'penghasilanBruto'
    | 'pphDipotong'
    | 'idTku'
    | 'negaraDipotong'
    | 'dokReferensi'
    | 'nomorDokumen'
    | 'created_by'
    | 'created_at'
    | 'updated_by'
    | 'updated_at'
    | 'internal_id'
    | 'keterangan1'
    | 'keterangan2'
    | 'keterangan3'
    | 'keterangan4'
    | 'keterangan5';
  valueOptions?: string[];
};

type TKodeObjekPajak = {
  kode: string;
  nama: string;
  pasal: string;
  statuspph: string;
  normanetto: string;
};

export function NrListView() {
  const apiRef = useGridApiRef();
  const navigate = useNavigate();

  const tableKey = 'nr';

  const { tables, filters, setPagination, resetPagination, setFilter } =
    usePaginationStore.getState();

  const [filterModel, setFilterModel] = useState<GridFilterModel>({
    items: filters[tableKey]?.items ?? [],
  });

  const [sortModel, setSortModel] = useState<GridSortModel>([]);
  const [rowSelectionModel, setRowSelectionModel] = useState<GridRowSelectionModel | undefined>(
    undefined
  );

  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState<boolean>(false);
  const [isUploadModalOpen, setIsUploadModalOpen] = useState<boolean>(false);
  const [isCancelModalOpen, setIsCancelModalOpen] = useState<boolean>(false);
  const [isPreviewOpen, setIsPreviewOpen] = useState<boolean>(false);
  const [previewPayload, setPreviewPayload] = useState<Record<string, any> | undefined>(undefined);

  const dataSelectedRef = useRef<any[]>([]);
  const [selectionVersion, setSelectionVersion] = useState(0);
  const [kodeObjekPajaks, setKodeObjekPajaks] = useState<TKodeObjekPajak[]>([]);
  const [localPagination, setLocalPagination] = useState({
    page: tables[tableKey]?.page ?? 0,
    pageSize: tables[tableKey]?.pageSize ?? 10,
  });

  const { data: kodeObjekPajak } = useGetKodeObjekPajak();
  const signer = useSelector((state: RootState) => state.user.data.signer);

  const { buildAdvancedFilter, buildRequestParams } = useAdvancedFilter();

  const page = tables[tableKey]?.page ?? 0;
  const pageSize = tables[tableKey]?.pageSize ?? 10;

  const isSyncingRef = useRef(false);
  const isEqual = (a: any, b: any) => a === b || JSON.stringify(a) === JSON.stringify(b);

  // 🔁 Sync store -> local
  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const newStoreItems = state.filters[tableKey]?.items ?? [];
      const localItems = filterModel.items ?? [];

      if (!isEqual(newStoreItems, localItems)) {
        isSyncingRef.current = true;
        setFilterModel({ items: newStoreItems });
        queueMicrotask(() => (isSyncingRef.current = false));
      }
    });

    return () => unsub();
  }, [filterModel.items]);

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const storePage = state.tables[tableKey]?.page ?? 0;
      const storePageSize = state.tables[tableKey]?.pageSize ?? 10;

      setLocalPagination((prev) =>
        prev.page !== storePage || prev.pageSize !== storePageSize
          ? { page: storePage, pageSize: storePageSize }
          : prev
      );
    });
    return () => unsub();
  }, []);

  // 🔁 Sync local -> store
  useEffect(() => {
    if (isSyncingRef.current) return;
    const currentStore = usePaginationStore.getState().filters[tableKey]?.items ?? [];
    if (!isEqual(currentStore, filterModel.items)) {
      setFilter(tableKey, { items: filterModel.items });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [filterModel]);

  const params = useMemo(() => {
    const advanced = buildAdvancedFilter(filterModel.items);

    const baseParams = {
      page,
      limit: pageSize,
      noBupot: '',
      idDipotong: '',
      namaDipotong: '',
      msPajak: '',
      thnPajak: '',
      sortingMode: sortModel[0]?.field ?? '',
      sortingMethod: sortModel[0]?.sort ?? '',
    };

    return buildRequestParams(baseParams, advanced);
  }, [page, pageSize, sortModel, filterModel.items, buildAdvancedFilter, buildRequestParams]);

  const { data, isFetching, refetch } = useGetNr({
    params,
  });
  const idStatusMapRef = useRef<Map<string | number, string>>(new Map());
  const rows = useMemo(() => (data?.data || []).filter(Boolean), [data?.data]);
  const totalRows = Number(data?.total ?? 0);

  useEffect(() => {
    try {
      const m = new Map<string | number, string>();
      (rows || []).forEach((r: any) => {
        const id = String(r.id ?? r.internal_id ?? '');
        m.set(id, r?.fgStatus ?? '');
      });
      // swap reference (cheap)
      idStatusMapRef.current = m;
    } catch {
      idStatusMapRef.current = new Map();
    }
  }, [rows]);

  const handlePaginationChange = (model: GridPaginationModel) => {
    // Update UI langsung (instan)
    setLocalPagination(model);

    // Sinkronisasi Zustand (tanpa delay visual)
    setPagination(tableKey, {
      page: model.page,
      pageSize: model.pageSize,
    });
  };

  const debouncedFilterChange = useDebounce((model: GridFilterModel) => {
    setFilterModel(model);
  }, 400);

  const debouncedSortChange = useDebounce((model: GridSortModel) => {
    setSortModel(model);
    resetPagination(tableKey);
  }, 400);

  // ---------- status options and columns (kept identical to your original) ----------
  type Status = 'draft' | 'normal' | 'cancelled' | 'amended';
  type StatusOption = { value: Status; label: string };
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const statusOptions: StatusOption[] = [
    { value: 'draft', label: 'Draft' },
    { value: 'normal', label: 'Normal' },
    { value: 'cancelled', label: 'Dibatalkan' },
    { value: 'amended', label: 'Normal Pengganti' },
  ];

  const columns = useMemo<IColumnGrid[]>(
    () => [
      {
        field: 'fgStatus',
        headerName: 'Status',
        width: 200,
        type: 'singleSelect',
        valueOptions: statusOptions.map((opt) => opt.value),
        valueFormatter: ({ value }: { value: string }) => {
          const option = statusOptions.find((opt) => opt.value === value);
          return option ? option.label : value;
        },
        renderCell: ({ value, row }) => <StatusChip value={value} revNo={row.revNo} />,
      },
      { field: 'noBupot', headerName: 'Nomor Bukti Pemotongan', width: 200 },
      { field: 'masaPajak', headerName: 'Masa Pajak', width: 150 },
      { field: 'tahunPajak', headerName: 'Tahun Pajak', width: 150 },
      { field: 'kodeObjekPajak', headerName: 'Kode Objek Pajak', width: 150 },
      { field: 'npwpPemotong', headerName: 'Identitas', width: 150 },
      { field: 'namaDipotong', headerName: 'Nama', width: 150 },
      {
        field: 'negaraDipotong',
        headerName: 'Negara',
        width: 180,
        renderCell: ({ row }) => {
          const kode = row.negaraDipotong || '-';
          const nama = row.namaNegara || '';
          return `${kode}${nama ? ' - ' + nama : ''}`;
        },
      },
      {
        field: 'penghasilanBruto',
        headerName: 'Jumlah Penghasilan Bruto (Rp)',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.penghasilanBruto),
      },
      {
        field: 'pphDipotong',
        headerName: 'Jumlah PPh Terutang (Rp)',
        width: 200,
        renderCell: ({ row }) => formatRupiah(row.pphDipotong),
      },
      { field: 'idTku', headerName: 'NITKU Pemotong', width: 150 },
      { field: 'dokReferensi', headerName: 'Nama dokumen', width: 150 },
      { field: 'nomorDokumen', headerName: 'Nomor dokumen', width: 150 },
      { field: 'created_by', headerName: 'Created', width: 150 },
      { field: 'created_at', headerName: 'Created At', width: 150 },
      { field: 'updated_by', headerName: 'Updated', width: 150 },
      { field: 'updated_at', headerName: 'Update At', width: 150 },
      { field: 'internal_id', headerName: 'Referensi', width: 150 },
      { field: 'keterangan1', headerName: 'Keterangan 1', width: 150 },
      { field: 'keterangan2', headerName: 'Keterangan 2', width: 200 },
      { field: 'keterangan3', headerName: 'Keterangan 3', width: 150 },
      { field: 'keterangan4', headerName: 'Keterangan 4', width: 150 },
      { field: 'keterangan5', headerName: 'Keterangan 5', width: 150 },
    ],
    [statusOptions]
  );

  const getSelectedRowByKey = (key?: GridRowId | 'all') => {
    const api = apiRef.current;
    if (!api) return null;

    const selectedRows = Array.from(api.getSelectedRows().values());
    if (key === 'all') return selectedRows;
    return selectedRows[0] ?? null;
  };

  const handleEditData = useCallback(
    (type = 'ubah') => {
      const selectedRow = dataSelectedRef.current[0];
      if (!selectedRow) return;
      navigate(`/unifikasi/nr/${selectedRow.id}/${type}`);
    },
    [navigate]
  );

  const throttledSelectionChange = useThrottle((newSelection: any) => {
    if (!apiRef.current) return;
    const ids =
      newSelection?.ids instanceof Set ? Array.from(newSelection.ids) : newSelection || [];
    const selectedData = ids.map((id: any) => apiRef.current!.getRow(id)).filter(Boolean);

    unstable_batchedUpdates(() => {
      dataSelectedRef.current = selectedData;
      setRowSelectionModel(newSelection);
      setSelectionVersion((v) => v + 1);
    });
  }, 150);

  // auto sync selected after rows set (schedule as idle to avoid heavy main thread work)
  useEffect(() => {
    const api = apiRef.current;
    if (!api) return;
    const unsubscribe = api.subscribeEvent('rowsSet', () => {
      const exec = () => {
        const ids =
          api.state?.rowSelection?.ids instanceof Set ? Array.from(api.state.rowSelection.ids) : [];
        const updatedSelected = ids.map((id) => api.getRow(id)).filter(Boolean);
        dataSelectedRef.current = updatedSelected;
        setSelectionVersion((v) => v + 1);
      };
      if ((window as any).requestIdleCallback) (window as any).requestIdleCallback(exec);
      else setTimeout(exec, 0);
    });
    // eslint-disable-next-line consistent-return
    return () => unsubscribe();
  }, [apiRef]);

  // ---------- memoized toolbar validation (avoid recompute heavy every click) ----------
  const validatedActions = useMemo(() => {
    const dataSelected = dataSelectedRef.current;
    const count = dataSelected.length;
    const hasSelection = count > 0;
    if (!hasSelection) {
      return {
        canDetail: false,
        canEdit: false,
        canDelete: false,
        canUpload: false,
        canReplacement: false,
        canCancel: false,
      };
    }
    const allDraft = dataSelected.every((d) => d.fgStatus === FG_STATUS_DN.DRAFT);
    const allNormal = dataSelected.every((d) => d.fgStatus === FG_STATUS_DN.NORMAL_DONE);
    return {
      canDetail: count === 1,
      canEdit: count === 1 && allDraft,
      canDelete: hasSelection && allDraft,
      canUpload: hasSelection && allDraft,
      canReplacement: count === 1 && dataSelected[0].fgStatus === FG_STATUS_DN.NORMAL_DONE,
      canCancel: hasSelection && allNormal,
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [selectionVersion]);

  useEffect(() => {
    if (kodeObjekPajak?.data && Array.isArray(kodeObjekPajak.data)) {
      setKodeObjekPajaks(kodeObjekPajak.data);
    }
  }, [kodeObjekPajak]);

  // ✅ Handler tombol “Detail”
  const handleOpenPreview = () => {
    const selectedRow = dataSelectedRef.current?.[0];
    if (!selectedRow) {
      enqueueSnackbar('Pilih 1 baris untuk melihat detail', { variant: 'warning' });
      return;
    }

    console.log('🧾 selectedRow:', selectedRow);
    console.log('🧩 Keys:', Object.keys(selectedRow));

    const kode = selectedRow.kodeObjekPajak || selectedRow.kdObjPjk;
    const detailKop = kodeObjekPajaks.find((item) => item.kode === kode);

    console.log(detailKop);
    const mergedRow = {
      ...selectedRow,
      ...(detailKop
        ? {
            namaObjekPajak: detailKop.nama,
            pasalPPh: detailKop.pasal,
            statusPPh: detailKop.statuspph,
            normaPenghasilanNeto: detailKop.normanetto,
          }
        : {}),
      tinDipotong: selectedRow.npwpPemotong ?? '',
      namaPenandatangan: signer,
    };

    setPreviewPayload(mergedRow);
    setIsPreviewOpen(true);
  };

  const actions = useMemo(
    () => [
      [
        {
          title: 'Refresh List',
          icon: <AutorenewTwoTone sx={{ width: 26, height: 26 }} />,
          func: () =>
            startTransition(() => {
              resetPagination(tableKey);
              void refetch();
            }),
        },
        {
          title: 'Edit',
          icon: <EditNoteTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => handleEditData('ubah'),
          disabled: !validatedActions.canEdit,
        },
        {
          title: 'Detail',
          icon: <ArticleTwoTone sx={{ width: 26, height: 26 }} />,
          func: handleOpenPreview,

          disabled: !validatedActions.canDetail,
        },
        {
          title: 'Hapus',
          icon: <DeleteSweepTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsDeleteModalOpen(true),
          disabled: !validatedActions.canDelete,
        },
      ],
      [
        {
          title: 'Upload',
          icon: <UploadFileTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsUploadModalOpen(true),
          disabled: !validatedActions.canUpload,
        },
        {
          title: 'Pengganti',
          icon: <FileOpenTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => handleEditData('pengganti'),
          disabled: !validatedActions.canReplacement,
        },
        {
          title: 'Batal',
          icon: <HighlightOffTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsCancelModalOpen(true),
          disabled: !validatedActions.canCancel,
        },
      ],
    ],
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [validatedActions, refetch, handleEditData]
  );

  const pinnedColumns = useMemo(() => ({ left: ['__check__', 'fgStatus', 'noBupot'] }), []);

  // Wrap CustomToolbar so types match GridToolbarProps and we can pass extra props
  const ToolbarWrapper: React.FC<GridToolbarProps> = useCallback(
    (gridToolbarProps) => (
      <CustomToolbar
        actions={actions}
        columns={columns}
        filterModel={filterModel}
        setFilterModel={setFilterModel}
        statusOptions={statusOptions}
        {...gridToolbarProps}
      />
    ),
    [actions, columns, filterModel, setFilterModel, statusOptions]
  );

  // schedule a lightweight debug (avoid console.debug in fast path)
  useEffect(() => {
    const api = apiRef.current;
    if (!api) return;
    const id = window.setTimeout(() => {
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
      const selected = getSelectedRowByKey('all');
    }, 100);
    // eslint-disable-next-line consistent-return
    return () => clearTimeout(id);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [apiRef, selectionVersion]);

  return (
    <>
      <DashboardContent>
        <CustomBreadcrumbs
          heading="Bupot Unifikasi Non Residen"
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'e-Bupot Unifikasi Non Residen' },
          ]}
          action={
            <Button component={RouterLink} to={paths.unifikasi.nrNew} variant="contained">
              Rekam Data
            </Button>
          }
        />

        <DataGridPremium
          apiRef={apiRef}
          checkboxSelection
          rows={rows || []}
          getRowId={(row: any) => {
            if (!row) return '';
            return String(row.id ?? row.internal_id ?? '');
          }}
          columns={columns}
          loading={isFetching}
          rowCount={totalRows}
          pagination
          paginationMode="server"
          paginationModel={localPagination}
          initialState={{
            pagination: {
              paginationModel: {
                page,
                pageSize,
              },
            },
          }}
          onPaginationModelChange={handlePaginationChange}
          pageSizeOptions={[5, 10, 15, 25, 50, 100]}
          filterMode="server"
          onFilterModelChange={debouncedFilterChange}
          sortingMode="server"
          onSortModelChange={debouncedSortChange}
          rowSelectionModel={rowSelectionModel}
          onRowSelectionModelChange={throttledSelectionChange}
          pinnedColumns={pinnedColumns}
          cellSelection
          showToolbar
          slots={{ toolbar: ToolbarWrapper }}
          sx={{
            border: 1,
            borderColor: 'divider',
            borderRadius: 2,
            mt: 3,
            '& .MuiDataGrid-cell': {
              borderColor: 'divider',
              userSelect: 'text',
              cursor: 'text',
            },
            '& .MuiDataGrid-columnHeaders': { borderColor: 'divider' },
          }}
        />
      </DashboardContent>

      {isDeleteModalOpen && (
        <ModalDeleteNr
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogDelete={isDeleteModalOpen}
          setIsOpenDialogDelete={setIsDeleteModalOpen}
          successMessage="Data berhasil dihapus"
        />
      )}

      {isUploadModalOpen && (
        <ModalUploadNr
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogUpload={isUploadModalOpen}
          setIsOpenDialogUpload={setIsUploadModalOpen}
          successMessage="Data berhasil diupload"
        />
      )}

      {isCancelModalOpen && (
        <ModalCancelNr
          dataSelected={dataSelectedRef.current}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogCancel={isCancelModalOpen}
          setIsOpenDialogCancel={setIsCancelModalOpen}
          successMessage="Data berhasil diupload"
        />
      )}

      {isPreviewOpen && (
        <ModalCetakPdfNr
          payload={previewPayload}
          isOpen={isPreviewOpen}
          onClose={() => {
            setIsPreviewOpen(false);
            setPreviewPayload(undefined);
          }}
        />
      )}
    </>
  );
}
