import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
import { paths } from 'src/routes/paths';
import React, { useCallback, useEffect, useMemo, useRef, useState, startTransition } from 'react';
import { useNavigate } from 'react-router';
import type {
  GridColDef,
  GridFilterModel,
  GridRowSelectionModel,
  GridSortModel,
  GridToolbarProps,
  GridPaginationModel,
} from '@mui/x-data-grid-premium';
import { DataGridPremium, useGridApiRef } from '@mui/x-data-grid-premium';
import { unstable_batchedUpdates } from 'react-dom';

import {
  ArticleTwoTone,
  AutorenewTwoTone,
  DeleteSweepTwoTone,
  EditNoteTwoTone,
  HighlightOffTwoTone,
  UploadFileTwoTone,
} from '@mui/icons-material';
import { enqueueSnackbar } from 'notistack';
import { usePaginationStore } from '../store/paginationStore';
import StatusChip from '../components/StatusChip';
import { useDebounce, useThrottle } from 'src/shared/hooks/useDebounceThrottle';
import useAdvancedFilter from '../hooks/useAdvancedFilterFakturReturPM';
import { CustomToolbar } from '../components/CustomToolbar';
import { useGetReturFakturPM, formatDateToDDMMYYYY } from '../hooks/useGetReturFakturPM';
import { formatRupiah } from 'src/shared/FormatRupiah/FormatRupiah';
import ModalDeleteReturPM from '../components/dialog/ModalDeleteReturPM';
import ModalCancelReturPM from '../components/dialog/ModalCancelReturPM';
import ModalUploadReturPM from '../components/dialog/ModalUploadReturPM';
import ModalCetakReturPM from '../components/dialog/ModalCetakFakturPM';

export type IColumnGrid = GridColDef & {
  field:
    | 'statusretur'
    | 'nomorretur'
    | 'nomorfakturdiretur'
    | 'npwppembeli'
    | 'tanggalretur'
    | 'masapajakretur'
    | 'tahunpajakretur'
    | 'nilaireturdpp'
    | 'nilaireturdpplain'
    | 'nilaireturdiskon'
    | 'nilaireturppn'
    | 'nilaireturppnbm'
    | 'created_by'
    | 'created_at'
    | 'updated_by'
    | 'updated_at';
  valueOptions?: readonly (string | number | { value: string | number; label: string })[];
};

export function ReturFakturPmListView() {
  const apiRef = useGridApiRef();
  const navigate = useNavigate();

  const tableKey = 'retur-faktur-pk';

  const { tables, filters, setPagination, resetPagination, setFilter } =
    usePaginationStore.getState();

  const [filterModel, setFilterModel] = useState<GridFilterModel>({
    items: filters[tableKey]?.items ?? [],
  });

  const [sortModel, setSortModel] = useState<GridSortModel>([]);
  const [rowSelectionModel, setRowSelectionModel] = useState<GridRowSelectionModel | undefined>(
    undefined
  );

  const [isUploadModalOpen, setIsUploadModalOpen] = useState<boolean>(false);

  const [isPreviewOpen, setIsPreviewOpen] = useState<boolean>(false);
  const [isCancelModalOpen, setIsCancelModalOpen] = useState<boolean>(false);
  const [previewPayload, setPreviewPayload] = useState<Record<string, any> | undefined>(undefined);

  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState<boolean>(false);

  const dataSelectedRef = useRef<any[]>([]);
  const [selectionVersion, setSelectionVersion] = useState(0);
  const [localPagination, setLocalPagination] = useState({
    page: tables[tableKey]?.page ?? 0,
    pageSize: tables[tableKey]?.pageSize ?? 10,
  });

  // const signer = useSelector((state: RootState) => state.user.data.signer);

  const { buildAdvancedFilter, buildRequestParams } = useAdvancedFilter();

  const page = tables[tableKey]?.page ?? 0;
  const pageSize = tables[tableKey]?.pageSize ?? 10;

  const isSyncingRef = useRef(false);
  const isEqual = (a: any, b: any) => a === b || JSON.stringify(a) === JSON.stringify(b);

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const newStoreItems = state.filters[tableKey]?.items ?? [];
      const localItems = filterModel.items ?? [];

      if (!isEqual(newStoreItems, localItems)) {
        isSyncingRef.current = true;
        setFilterModel({ items: newStoreItems });
        queueMicrotask(() => (isSyncingRef.current = false));
      }
    });

    return () => unsub();
  }, [filterModel.items]);

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const storePage = state.tables[tableKey]?.page ?? 0;
      const storePageSize = state.tables[tableKey]?.pageSize ?? 10;

      setLocalPagination((prev) =>
        prev.page !== storePage || prev.pageSize !== storePageSize
          ? { page: storePage, pageSize: storePageSize }
          : prev
      );
    });
    return () => unsub();
  }, []);

  useEffect(() => {
    if (isSyncingRef.current) return;
    const currentStore = usePaginationStore.getState().filters[tableKey]?.items ?? [];
    if (!isEqual(currentStore, filterModel.items)) {
      setFilter(tableKey, { items: filterModel.items });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [filterModel]);

  const params = useMemo(() => {
    const advanced = buildAdvancedFilter(filterModel.items);

    const baseParams = {
      page,
      limit: pageSize,
      sortingMode: sortModel[0]?.field ?? '',
      sortingMethod: sortModel[0]?.sort ?? '',
    };

    return buildRequestParams(baseParams, advanced);
  }, [page, pageSize, sortModel, filterModel.items, buildAdvancedFilter, buildRequestParams]);

  const { data, isFetching, refetch } = useGetReturFakturPM({
    params,
  });
  const idStatusMapRef = useRef<Map<string | number, string>>(new Map());
  const rows = useMemo(() => (data?.data || []).filter(Boolean), [data?.data]);
  const totalRows = Number(data?.total ?? 0);

  useEffect(() => {
    try {
      const m = new Map<string | number, string>();
      (rows || []).forEach((r: any) => {
        const id = String(r.id ?? r.internal_id ?? '');
        m.set(id, r?.fgStatus ?? '');
      });
      idStatusMapRef.current = m;
    } catch {
      idStatusMapRef.current = new Map();
    }
  }, [rows]);

  const handlePaginationChange = (model: GridPaginationModel) => {
    setLocalPagination(model);
    setPagination(tableKey, {
      page: model.page,
      pageSize: model.pageSize,
    });
  };

  const debouncedFilterChange = useDebounce((model: GridFilterModel) => {
    setFilterModel(model);
  }, 400);

  const debouncedSortChange = useDebounce((model: GridSortModel) => {
    setSortModel(model);
    resetPagination(tableKey);
  }, 400);

  type Status =
    | 'DRAFT'
    | 'APPROVED'
    | 'WAITING FOR AMENDMENT'
    | 'AMENDED'
    | 'WAITING FOR CANCELLATION';
  type StatusOption = { value: Status; label: string };
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const statusOptions: StatusOption[] = [
    { value: 'DRAFT', label: 'Draft' },
    { value: 'APPROVED', label: 'Normal' },
    { value: 'WAITING FOR AMENDMENT', label: 'Normal Pengganti' },
    { value: 'AMENDED', label: 'Diganti' },
    { value: 'WAITING FOR CANCELLATION', label: 'Batal' },
  ];

  const columns = useMemo<IColumnGrid[]>(
    () => [
      {
        field: 'statusretur',
        headerName: 'Status',
        width: 200,
        type: 'singleSelect',
        valueOptions: statusOptions.map((opt) => opt.value),
        valueFormatter: ({ value }: { value: string }) => {
          const option = statusOptions.find((opt) => opt.value === value);
          console.log(option);
          return option ? option.label : value;
        },
        renderCell: ({ value, row }) => (
          <StatusChip value={value} credit={row.statuspembeli} valid={row.buyerstatus} />
        ),
      },
      { field: 'nomorretur', headerName: 'Nomor Retur Pajak', width: 200 },
      {
        field: 'nomorfakturdiretur',
        headerName: 'Nomor Faktur Pajak',
        width: 200,
      },
      {
        field: 'npwppembeli',
        headerName: 'NPWP',
        width: 200,
      },
      {
        field: 'tanggalretur',
        headerName: 'Tanggal Retur',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.tanggalretur),
      },
      {
        field: 'masapajakretur',
        headerName: 'Masa',
        width: 150,
      },
      {
        field: 'tahunpajakretur',
        headerName: 'Tahun',
        width: 150,
      },
      {
        field: 'nilaireturdpp',
        headerName: 'Retur DPP',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturdpp),
      },
      {
        field: 'nilaireturdpplain',
        headerName: 'Retur DPP Nilai Lain',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturdpplain),
      },
      {
        field: 'nilaireturdiskon',
        headerName: 'Retur Diskon',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturdiskon),
      },
      {
        field: 'nilaireturppn',
        headerName: 'Retur PPN',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturppn),
      },
      {
        field: 'nilaireturppnbm',
        headerName: 'Retur PPnBM',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturppnbm),
      },
      {
        field: 'created_by',
        headerName: 'User Perekam',
        width: 150,
      },
      {
        field: 'created_at',
        headerName: 'Tanggal Rekam',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.created_at),
      },

      {
        field: 'updated_by',
        headerName: 'User Pengubah',
        width: 150,
      },
      {
        field: 'updated_at',
        headerName: 'Tanggal Ubah',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.updated_at),
      },
    ],
    [statusOptions]
  );

  const handleEditData = useCallback(
    (type = 'ubah') => {
      const selectedRow = dataSelectedRef.current[0];
      if (!selectedRow) return;
      navigate(`/faktur/retur-pm/${selectedRow.id}/${type}`);
    },
    [navigate]
  );

  const throttledSelectionChange = useThrottle((newSelection: any) => {
    if (!apiRef.current) return;
    const ids =
      newSelection?.ids instanceof Set ? Array.from(newSelection.ids) : newSelection || [];
    const selectedData = ids.map((id: any) => apiRef.current!.getRow(id)).filter(Boolean);

    unstable_batchedUpdates(() => {
      dataSelectedRef.current = selectedData;
      setRowSelectionModel(newSelection);
      setSelectionVersion((v) => v + 1);
    });
  }, 150);

  useEffect(() => {
    const api = apiRef.current;
    if (!api) return;
    const unsubscribe = api.subscribeEvent('rowsSet', () => {
      const exec = () => {
        const ids =
          api.state?.rowSelection?.ids instanceof Set ? Array.from(api.state.rowSelection.ids) : [];
        const updatedSelected = ids.map((id) => api.getRow(id)).filter(Boolean);
        dataSelectedRef.current = updatedSelected;
        setSelectionVersion((v) => v + 1);
      };
      if ((window as any).requestIdleCallback) (window as any).requestIdleCallback(exec);
      else setTimeout(exec, 0);
    });
    // eslint-disable-next-line consistent-return
    return () => unsubscribe();
  }, [apiRef]);

  const validatedActions = useMemo(() => {
    const dataSelected = dataSelectedRef.current;
    const count = dataSelected.length;

    if (count === 0) {
      return {
        canDetail: false,
        canChange: false,
        canCancel: false,
        canDelete: false,
        canUpload: false,
      };
    }

    const isSingle = count === 1;

    const allDraft = dataSelected.every((d) => String(d.statusretur).toUpperCase() === 'DRAFT');

    const allApproved = dataSelected.every(
      (d) => String(d.statusretur).toUpperCase() === 'APPROVED'
    );

    return {
      canDetail: isSingle,
      canChange: isSingle && allDraft,
      canCancel: allApproved,
      canDelete: allDraft,
      canUpload: allDraft,
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [selectionVersion]);

  const handleOpenPreview = () => {
    const selectedRow = dataSelectedRef.current?.[0];
    if (!selectedRow) {
      enqueueSnackbar('Pilih 1 baris untuk melihat detail', { variant: 'warning' });
      return;
    }

    console.log(selectedRow);

    setPreviewPayload(selectedRow);
    setIsPreviewOpen(true);
  };

  const actions = useMemo(
    () => [
      [
        {
          title: 'Refresh List',
          icon: <AutorenewTwoTone sx={{ width: 26, height: 26 }} />,
          func: () =>
            startTransition(() => {
              resetPagination(tableKey);
              void refetch();
            }),
        },
        {
          title: 'Detail',
          icon: <ArticleTwoTone sx={{ width: 26, height: 26 }} />,
          func: handleOpenPreview,
          disabled: !validatedActions.canDetail,
        },
      ],
      [
        {
          title: 'Ubah',
          icon: <EditNoteTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => handleEditData('ubah'),
          disabled: !validatedActions.canChange,
        },
        {
          title: 'Hapus',
          icon: <DeleteSweepTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsDeleteModalOpen(true),
          disabled: !validatedActions.canDelete,
        },
        {
          title: 'Batal',
          icon: <HighlightOffTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsCancelModalOpen(true),
          disabled: !validatedActions.canCancel,
        },
      ],
      [
        {
          title: 'Upload',
          icon: <UploadFileTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsUploadModalOpen(true),
          disabled: !validatedActions.canUpload,
        },
      ],
    ],
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [validatedActions, refetch, handleEditData]
  );

  const pinnedColumns = useMemo(
    () => ({ left: ['__check__', 'statusretur', 'nomorretur', 'nomorfakturdiretur'] }),
    []
  );

  const ToolbarWrapper: React.FC<GridToolbarProps> = useCallback(
    (gridToolbarProps) => (
      <CustomToolbar
        actions={actions}
        columns={columns}
        filterModel={filterModel}
        setFilterModel={setFilterModel}
        statusOptions={statusOptions}
        {...gridToolbarProps}
      />
    ),
    [actions, columns, filterModel, setFilterModel, statusOptions]
  );

  return (
    <>
      <DashboardContent>
        <CustomBreadcrumbs
          heading="Retur Pajak Masukan"
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'Retur Pajak Masukan' },
          ]}
          // action={
          //   <Button variant="contained" onClick={() => setIsPrepopulatedOpen(true)}>
          //     Prepopulated
          //   </Button>
          // }
        />

        <DataGridPremium
          apiRef={apiRef}
          checkboxSelection
          rows={rows || []}
          getRowId={(row: any) => {
            if (!row) return '';
            return String(row.id ?? row.internal_id ?? '');
          }}
          columns={columns}
          loading={isFetching}
          rowCount={totalRows}
          pagination
          paginationMode="server"
          paginationModel={localPagination}
          initialState={{
            pagination: {
              paginationModel: {
                page,
                pageSize,
              },
            },
            columns: {
              columnVisibilityModel: {
                jenisInvoice: true, // hide on load
              },
            },
          }}
          onPaginationModelChange={handlePaginationChange}
          pageSizeOptions={[5, 10, 15, 25, 50, 100]}
          filterMode="server"
          onFilterModelChange={debouncedFilterChange}
          sortingMode="server"
          onSortModelChange={debouncedSortChange}
          rowSelectionModel={rowSelectionModel}
          onRowSelectionModelChange={throttledSelectionChange}
          pinnedColumns={pinnedColumns}
          cellSelection
          showToolbar
          slots={{ toolbar: ToolbarWrapper }}
          sx={{
            border: 1,
            borderColor: 'divider',
            borderRadius: 2,
            mt: 3,
            '& .MuiDataGrid-cell': {
              borderColor: 'divider',
              userSelect: 'text',
              cursor: 'text',
            },
            '& .MuiDataGrid-columnHeaders': { borderColor: 'divider' },
          }}
          onColumnVisibilityModelChange={(newModel) => {
            if (newModel['__check__'] === false) {
              newModel['__check__'] = true;
            }
            apiRef.current?.setColumnVisibilityModel(newModel);
          }}
          disableVirtualization
        />
      </DashboardContent>

      {isUploadModalOpen && (
        <ModalUploadReturPM
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogUpload={isUploadModalOpen}
          setIsOpenDialogUpload={setIsUploadModalOpen}
          successMessage="Data berhasil diupload"
          onConfirmUpload={async () => {
            await refetch();
          }}
        />
      )}
      {isCancelModalOpen && (
        <ModalCancelReturPM
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogCancel={isCancelModalOpen}
          setIsOpenDialogCancel={setIsCancelModalOpen}
          successMessage="Data berhasil dihapus"
          onConfirmCancel={async () => {
            await refetch();
          }}
        />
      )}

      {isPreviewOpen && (
        <ModalCetakReturPM
          payload={previewPayload}
          isOpen={isPreviewOpen}
          onClose={() => {
            setIsPreviewOpen(false);
            setPreviewPayload(undefined);
          }}
        />
      )}

      {isDeleteModalOpen && (
        <ModalDeleteReturPM
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogDelete={isDeleteModalOpen}
          setIsOpenDialogDelete={setIsDeleteModalOpen}
          successMessage="Data berhasil dihapus"
          onConfirmDelete={async () => {
            await refetch();
          }}
        />
      )}
    </>
  );
}
