import { useMutation } from '@tanstack/react-query';
import dayjs from 'dayjs';
import type { TPostDokumenLainKeluaranRequest } from '../types/types';
import fakturApi from '../utils/api';

/**
 * 🔒 Helper normalisasi tanggal
 * - Date / Dayjs  → format DDMMYYYY
 * - string 8 digit → pakai langsung
 */
const normalizeTanggal = (val: any): string => {
  if (!val) return dayjs().format('DDMMYYYY');

  // sudah string DDMMYYYY → JANGAN diparse ulang
  if (typeof val === 'string' && /^\d{8}$/.test(val)) {
    return val;
  }

  // Date / Dayjs
  return dayjs(val).format('DDMMYYYY');
};

/**
 * Transformasi data form → payload Dokumen Lain Keluaran
 */
const transformParams = (formData: any): TPostDokumenLainKeluaranRequest => {
  const {
    id,
    // isCreatedByBuyer,
    kdTransaksi,

    detailTransaksi,
    keteranganTambahan,

    dokumenTransaksi,
    kodeDokumen,
    nomorDokumen,
    nomorDokumenDiganti,

    tanggalDokumen,
    tanggalDokumenDiganti,

    masaPajak,
    tahunPajak,
    referensi,

    jumlahDpp,
    jumlahPpn,
    jumlahPpnbm,

    objekFaktur,

    namaPembeli,
    alamatPembeli,
    npwpPembeli,
    keterangan1,
    keterangan2,
    keterangan3,
    keterangan4,
    keterangan5,
  } = formData;

  const isPengganti = Boolean(nomorDokumenDiganti);

  return {
    id: id ?? null,
    isCreatedByBuyer: false,
    kdTransaksi: kdTransaksi ?? '',
    detailTransaksi: detailTransaksi ?? '',
    keteranganTambahan: keteranganTambahan ?? null,

    fgPengganti: isPengganti ? 'TD.00401' : 'TD.00400',

    dokumenTransaksi: dokumenTransaksi ?? '',
    kodeDokumen: kodeDokumen ?? '',
    nomorDokumen: nomorDokumen ?? '',
    nomorDokumenDiganti: nomorDokumenDiganti ?? '',

    // ✅ FIX UTAMA ADA DI SINI
    tanggalDokumen: normalizeTanggal(tanggalDokumen),

    tanggalDokumenDiganti: tanggalDokumenDiganti ? normalizeTanggal(tanggalDokumenDiganti) : null,

    masaPajak: String(masaPajak ?? dayjs().month() + 1).padStart(2, '0'),
    tahunPajak: String(tahunPajak ?? dayjs().year()),
    referensi: referensi ?? '',

    jumlahDpp: Number(jumlahDpp ?? 0),
    jumlahPpn: Number(jumlahPpn ?? 0),
    jumlahPpnbm: Number(jumlahPpnbm ?? 0),

    objekFaktur:
      detailTransaksi === 'TD.00312' || detailTransaksi === 'TD.00313'
        ? (objekFaktur ?? []).map((item: any) => ({
            brgJasa: item.brgJasa,
            kdBrgJasa: item.kdBrgJasa,
            namaBrgJasa: item.namaBrgJasa,
            satuanBrgJasa: item.satuanBrgJasa,

            hargaSatuan: String(item.hargaSatuan ?? '0'),
            jmlBrgJasa: String(item.jmlBrgJasa ?? '0'),
            totalHarga: String(item.totalHarga ?? '0'),
            diskon: String(item.diskon ?? '0'),

            dpp: String(item.dpp ?? '0'),
            tarifPpn: String(item.tarifPpn ?? '0'),
            ppn: Number(item.ppn ?? 0),

            tarifPpnbm: String(item.tarifPpnbm ?? '0'),
            ppnbm: String(item.ppnbm ?? '0'),

            cekDppLain: item.cekDppLain === true,
          }))
        : [],

    namaPembeli: namaPembeli ?? '',
    alamatPembeli: alamatPembeli ?? '',
    npwpPembeli: npwpPembeli ?? '0000000000000000',
    keterangan1: keterangan1 ?? '',
    keterangan2: keterangan2 ?? '',
    keterangan3: keterangan3 ?? '',
    keterangan4: keterangan4 ?? '',
    keterangan5: keterangan5 ?? '',
  };
};

/**
 * Hook simpan Dokumen Lain Keluaran
 */
const useSaveDokumenLainKeluaran = (props?: any) =>
  useMutation({
    mutationKey: ['save-dokumen-lain-keluaran'],
    mutationFn: async (params: any) => {
      const payload = transformParams(params);
      return fakturApi.saveDokumenLainKeluaran(payload);
    },
    ...props,
  });

export default useSaveDokumenLainKeluaran;
