import { useMutation } from '@tanstack/react-query';
import dayjs from 'dayjs';
import type { TPostFakturPKRequest } from '../types/types';
import fakturApi from '../utils/api';

/**
 * Mapping nilai radio ke boolean flag fgUangMuka & fgPelunasan
 */
const getPaymentFlags = (value: string) => {
  switch (value) {
    case 'uang_muka':
      return { fgUangMuka: true, fgPelunasan: false };
    case 'pelunasan':
      return { fgUangMuka: false, fgPelunasan: true };
    default:
      // single_payment atau lainnya
      return { fgUangMuka: false, fgPelunasan: false };
  }
};

/**
 * Transformasi data form menjadi payload untuk endpoint IF_TXR_001/create
 */
const transformParams = (formData: any): TPostFakturPKRequest => {
  const {
    id, // internal id
    fgUangMuka, // string dari radio
    nomorFaktur,
    nomorFakturDiganti,
    detailTransaksi,
    idKeteranganTambahan,
    keteranganTambahan,
    masaPajak,
    tahunPajak,
    refDoc,
    referensi,
    npwpPembeli,
    idLainPembeli,
    kdNegaraPembeli,
    nikPaspPembeli,
    namaPembeli,
    tkuPembeli,
    alamatPembeli,
    emailPembeli,
    keterangan1,
    keterangan2,
    keterangan3,
    keterangan4,
    keterangan5,
    objekFaktur,
    jumlahUangMuka,
    totalDpp,
    totalDppLain,
    totalPpn,
    totalPpnbm,
    tanggalFaktur,
    fgPengganti,
    capKetTambahan,
    uangMukaDpp,
    uangMukaDppLain,
    uangMukaPpn,
    uangMukaPpnbm,
  } = formData;

  let finalNpwpPembeli = npwpPembeli;
  let finalNikPaspPembeli = nikPaspPembeli;

  if (idLainPembeli === '2' || idLainPembeli === '3') {
    // jika Paspor / ID Lain → nikPaspPembeli ambil dari npwpPembeli
    finalNikPaspPembeli = npwpPembeli || '';

    // npwpPembeli HARUS menjadi 16 digit nol
    finalNpwpPembeli = '0000000000000000';
  }

  // 🔁 Konversi nilai radio ke boolean flag
  const { fgUangMuka: flagUangMuka, fgPelunasan: flagPelunasan } = getPaymentFlags(fgUangMuka);

  const totalDiskon = Array.isArray(objekFaktur)
    ? objekFaktur.reduce((sum, item) => sum + Number(item.diskon ?? 0), 0)
    : 0;

  return {
    id: id ?? null,
    fgUangMuka: flagUangMuka,
    fgPelunasan: flagPelunasan,
    nomorFaktur: nomorFaktur ?? '',
    nomorFakturDiganti: nomorFakturDiganti ?? '',
    detailTransaksi: detailTransaksi ?? '',
    idKeteranganTambahan: idKeteranganTambahan ?? '',
    keteranganTambahan: keteranganTambahan ?? '',
    masaPajak: String(masaPajak ?? dayjs().month() + 1).padStart(2, '0'),
    tahunPajak: String(tahunPajak ?? dayjs().year()),
    refDoc: refDoc ?? '',
    referensi: referensi ?? '',

    // npwpPembeli: npwpPembeli ?? '',
    npwpPembeli: finalNpwpPembeli,
    idLainPembeli: idLainPembeli ?? '',
    kdNegaraPembeli: kdNegaraPembeli ?? 'IDN',
    // nikPaspPembeli: nikPaspPembeli ?? '',
    nikPaspPembeli: finalNikPaspPembeli,
    namaPembeli: namaPembeli ?? '',
    tkuPembeli: tkuPembeli ?? '',
    alamatPembeli: alamatPembeli ?? '',
    emailPembeli: emailPembeli ?? '',

    keterangan1: keterangan1 ?? '',
    keterangan2: keterangan2 ?? '',
    keterangan3: keterangan3 ?? '',
    keterangan4: keterangan4 ?? '',
    keterangan5: keterangan5 ?? '',

    objekFaktur: Array.isArray(objekFaktur)
      ? objekFaktur.map((item) => ({
          brgJasa: item.brgJasa ?? '',
          kdBrgJasa: item.kdBrgJasa ?? '',
          namaBrgJasa: item.namaBrgJasa ?? '',
          satuanBrgJasa: item.satuanBrgJasa ?? '',
          hargaSatuan: Number(item.hargaSatuan ?? 0),
          jmlBrgJasa: Number(item.jmlBrgJasa ?? 0),
          totalHarga: Number(item.totalHarga ?? 0),
          diskon: Number(item.diskon ?? 0),
          // cekDppLain: item.cekDppLain,
          cekDppLain: item.cekDppLain !== undefined ? item.cekDppLain : null, // biarkan kosong, jangan ubah

          dpp: Number(item.dpp ?? 0),
          dppLain: Number(item.dppLain ?? 0),
          tarifPpn: Number(item.tarifPpn ?? 11),
          ppn: Number(item.ppn ?? 0),
          tarifPpnbm: Number(item.tarifPpnbm ?? 0),
          ppnbm: Number(item.ppnbm ?? 0),
        }))
      : [],

    jumlahUangMuka:
      fgUangMuka === 'uang_muka'
        ? Number(jumlahUangMuka ?? 0)
        : fgUangMuka === 'pelunasan'
          ? Number(uangMukaDpp ?? 0)
          : 0,

    // jumlahUangMuka: Number(totalDpp ?? 0),
    totalDpp: Number(totalDpp ?? 0),
    totalDppLain: Number(totalDppLain ?? 0),
    totalPpn: Number(totalPpn ?? 0),
    totalPpnbm: Number(totalPpnbm ?? 0),

    uangMukaDpp: Number(uangMukaDpp ?? 0),
    uangMukaDppLain: Number(uangMukaDppLain ?? 0),
    uangMukaPpn: Number(uangMukaPpn ?? 0),
    uangMukaPpnbm: Number(uangMukaPpnbm ?? 0),

    tanggalFaktur: tanggalFaktur
      ? dayjs(String(tanggalFaktur)).format('DDMMYYYY')
      : dayjs().format('DDMMYYYY'),

    fgPengganti: String(fgPengganti ?? '0'),
    capKetTambahan: capKetTambahan ?? '',
    totalDiskon: String(totalDiskon),
  };
};

/**
 * Hook untuk menyimpan data Faktur PK ke endpoint IF_TXR_001/create
 */
const useSaveFakturPk = (props?: any) =>
  useMutation({
    mutationKey: ['save-fakturpk'],
    mutationFn: async (params: any) => {
      const payload = transformParams(params);
      return fakturApi.saveFakturPK(payload);
    },
    ...props,
  });

export default useSaveFakturPk;
