import { z } from 'zod';

export const BarangJasaSchema = z
  .object({
    // ======================
    // EXISTING (TIDAK DIUBAH)
    // ======================
    type: z.string().nonempty('Type wajib diisi'),
    kdBrgJasa: z.string().nonempty('Kode Barang Jasa wajib diisi'),
    namaBrgJasa: z.string().trim().nonempty('Nama Barang/Jasa wajib diisi'),
    satuanBrgJasa: z.string().nonempty('Satuan Barang Jasa wajib diisi'),

    jmlBrgJasa: z
      .string()
      .trim()
      .nonempty('Jumlah Barang wajib diisi')
      .refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
        message: 'Jumlah Barang wajib diisi dengan angka lebih dari 0',
      }),

    hargaSatuan: z.string().trim().nonempty('Harga Satuan wajib diisi'),

    totalHarga: z.string().trim().nonempty('Harga Total wajib diisi'),

    diskon: z
      .string()
      .trim()
      .nonempty('Diskon wajib diisi')
      .refine((val) => !isNaN(Number(val)) && Number(val) >= 0, {
        message: 'Diskon wajib diisi dengan angka valid',
      }),

    fgPmk: z.string().optional(),
    dpp: z.string().nonempty('DPP wajib diisi'),
    tarifPpn: z.string().nonempty('Tarif PPN wajib diisi'),

    dppLain: z
      .string()
      .min(1, 'DPP Nilai Lain wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'DPP Nilai Lain harus berupa angka' }),

    ppn: z
      .string()
      .min(1, 'PPN wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'PPN harus berupa angka' }),

    ppnbm: z
      .string()
      .min(1, 'PPnBM wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'PPnBM harus berupa angka' }),

    tarifPpnbm: z
      .string()
      .min(1, 'Tarif PPnBM wajib diisi')
      .refine((val) => !isNaN(Number(val)), { message: 'Tarif PPnBM harus berupa angka' }),

    // ======================
    // ➕ FIELD RETUR (BARU)
    // ======================
    returJmlBrgJasa: z
      .string()
      .optional()
      .refine((v) => !v || (!isNaN(Number(v)) && Number(v) >= 0), {
        message: 'Retur jumlah harus berupa angka valid',
      }),

    totalHargaRetur: z
      .string()
      .optional()
      .refine((v) => !v || !isNaN(Number(v)), {
        message: 'Total harga retur harus berupa angka',
      }),

    returDiskon: z
      .string()
      .optional()
      .refine((v) => !v || !isNaN(Number(v)), {
        message: 'Retur diskon harus berupa angka',
      }),

    returDpp: z
      .string()
      .optional()
      .refine((v) => !v || !isNaN(Number(v)), {
        message: 'Retur DPP harus berupa angka',
      }),

    returDppLain: z
      .string()
      .optional()
      .refine((v) => !v || !isNaN(Number(v)), {
        message: 'Retur DPP Nilai Lain harus berupa angka',
      }),

    returPpn: z
      .string()
      .optional()
      .refine((v) => !v || !isNaN(Number(v)), {
        message: 'Retur PPN harus berupa angka',
      }),

    returPpnbm: z
      .string()
      .optional()
      .refine((v) => !v || !isNaN(Number(v)), {
        message: 'Retur PPnBM harus berupa angka',
      }),
  })

  // ======================
  // EXISTING REFINE (TIDAK DIUBAH)
  // ======================
  .refine(
    (data) => {
      const hargaTotal = Number(data.totalHarga);
      const diskon = Number(data.diskon);
      if (isNaN(hargaTotal) || isNaN(diskon)) return true;
      return diskon <= hargaTotal;
    },
    {
      message: 'Diskon tidak boleh lebih dari Harga Total',
      path: ['diskon'],
    }
  )

  // ======================
  // EXISTING + RETUR SUPER REFINE
  // ======================
  .superRefine((data, ctx) => {
    const dpp = Number(data.dpp || 0);

    const fieldsToCheck = [
      { key: 'dppLain', label: 'DPP Nilai Lain' },
      { key: 'ppn', label: 'PPN' },
      { key: 'ppnbm', label: 'PPnBM' },
      { key: 'tarifPpnbm', label: 'Tarif PPnBM' },
    ] as const;

    fieldsToCheck.forEach(({ key, label }) => {
      const val = Number(data[key] || 0);
      if (!isNaN(val) && val > dpp) {
        ctx.addIssue({
          code: 'custom',
          path: [key],
          message: `${label} tidak boleh lebih dari jumlah DPP`,
        });
      }
    });

    // ======================
    // ➕ VALIDASI RETUR
    // ======================
    const jml = Number(data.jmlBrgJasa || 0);
    const retur = Number(data.returJmlBrgJasa || 0);

    if (retur > jml) {
      ctx.addIssue({
        code: 'custom',
        path: ['returJmlBrgJasa'],
        message: 'Jumlah retur tidak boleh lebih besar dari jumlah barang',
      });
    }

    const totalRetur = Number(data.totalHargaRetur || 0);
    const returDiskon = Number(data.returDiskon || 0);

    if (returDiskon > totalRetur) {
      ctx.addIssue({
        code: 'custom',
        path: ['returDiskon'],
        message: 'Diskon retur tidak boleh lebih besar dari total harga retur',
      });
    }
  });

export type TBarangJasaSchema = z.infer<typeof BarangJasaSchema>;
