import React, { useEffect, useMemo, useState } from 'react';
import { DataGridPremium, useGridApiRef } from '@mui/x-data-grid-premium';
import Stack from '@mui/material/Stack';
import type { GridColDef } from 'node_modules/@mui/x-data-grid/esm/models/colDef/gridColDef';
import IconButton from '@mui/material/IconButton';
import Grid from '@mui/material/Grid';
import Divider from '@mui/material/Divider';
import { Edit } from '@mui/icons-material';
import { useFormContext, useWatch } from 'react-hook-form';

// import ModalDetailTransaksiBarang from './dialog/ModalDetailTransaksiBarang';

// import useGetGoods from 'src/sections/faktur/fakturPk/hooks/useGetGoods';
import ModalDetailTransaksiBarang from '../dialog/ModalDetailTransaksiBarang';
import useGetServices from 'src/sections/faktur/fakturPk/hooks/useGetServices';
import useGetSatuan from 'src/sections/faktur/fakturPk/hooks/useGetSatuan';
import CustomNoRowsOverlay from './NoRowsOverlay';
import useGetGoods from '../../hooks/useGetGoods';
// import useGetGoods from '../../hooks/useGetGoods';

const rupiahFormat = (value: any) => new Intl.NumberFormat('id-ID', {}).format(Number(value));

export type IColumnGrid = GridColDef & {
  field:
    | 'actions'
    | 'brgJasa'
    | 'namaBrgJasa'
    | 'kdBrgJasa'
    | 'jmlBrgJasa'
    | 'satuanBrgJasa'
    | 'hargaSatuan'
    | 'totalHarga'
    | 'diskon'
    | 'dpp'
    | 'dppLain'
    | 'ppn'
    | 'ppnbm'
    | 'tarifPpnbm'
    | 'returJmlBrgJasa'
    | 'returDpp'
    | 'returDppLain'
    | 'returPpn'
    | 'returPpnbm';
};

type Props = {
  rekamData?: any | null;
  isLoading?: boolean;
};

const DetailTransaksi: React.FC<Props> = ({ rekamData = null, isLoading = false }) => {
  //   const { type } = useParams<{ id?: string; type?: 'ubah' | 'pengganti' | 'new' }>();
  const apiRef = useGridApiRef();
  const [openModal, setOpenModal] = useState<boolean>(false);

  const goodsQuery = useGetGoods();
  const servicesQuery = useGetServices();
  const satuanQuery = useGetSatuan();

  // data table
  const [rows, setRows] = useState<any[]>([]);
  const [editRow, setEditRow] = useState<any | null>(null);

  const { control, setValue, getValues } = useFormContext();
  const detailTransaksi = useWatch({ control, name: 'detailTransaksi' });

  // open modal edit
  const handleEdit = (params: any) => {
    setEditRow(params.row);
    setOpenModal(true);
  };

  const columns = useMemo<IColumnGrid[]>(
    () => [
      {
        field: 'actions',
        type: 'actions',
        headerName: 'Aksi',
        headerAlign: 'center',
        align: 'center',
        minWidth: 150,
        renderCell: (params) => (
          <Stack direction="row" gap={2}>
            <IconButton onClick={() => handleEdit(params)}>
              <Edit />
            </IconButton>
          </Stack>
        ),
      },
      {
        field: 'brgJasa',
        headerName: 'Type',
        minWidth: 200,
      },
      {
        field: 'namaBrgJasa',
        headerName: 'Nama',
        minWidth: 200,
      },
      {
        field: 'kdBrgJasa',
        headerName: 'Kode',
        minWidth: 200,
      },
      {
        field: 'jmlBrgJasa',
        headerName: 'Jumlah Barang',
        minWidth: 200,
      },
      {
        field: 'satuanBrgJasa',
        headerName: 'Satuan Barang/Jasa',
        minWidth: 200,
      },
      {
        field: 'hargaSatuan',
        headerName: 'Harga Satuan (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'totalHarga',
        headerName: 'Jumlah Total (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'diskon',
        headerName: 'Diskon (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'dpp',
        headerName: 'DPP (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params.value)}</>,
      },
      {
        field: 'dppLain',
        headerName: 'DPP Nilai Lain (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params.value)}</>,
      },
      {
        field: 'ppn',
        headerName: 'PPN (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'ppnbm',
        headerName: 'PPnBM (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'tarifPpnbm',
        headerName: 'Tarif PPnBM (%)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'returJmlBrgJasa',
        headerName: 'Retur Jumlah Barang/Jasa',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'returDpp',
        headerName: 'Retur DPP (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'returDppLain',
        headerName: 'Retur DPP Nilai Lain (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'returPpn',
        headerName: 'Retur PPN (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
      {
        field: 'returPpnbm',
        headerName: 'Retur PPnBM (Rp)',
        minWidth: 200,
        renderCell: (params) => <>{rupiahFormat(params?.value)}</>,
      },
    ],
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [rekamData]
  );

  const handleSaveFromModal = (entry: any) => {
    if (editRow) {
      // mode edit
      setRows((prev) => prev.map((row) => (row.id === editRow.id ? { ...row, ...entry } : row)));
      setEditRow(null);
    } else {
      // mode tambah baru
      const id = String(Date.now());
      setRows((prev) => [...prev, { id, ...entry }]);
    }
    setOpenModal(false);
  };

  // Sync totals -> setValue (existing effect)
  useEffect(() => {
    const totalJumlahBarangRaw = rows.reduce((sum, r) => sum + (Number(r.jmlBrgJasa) || 0), 0);
    const totalHargaRaw = rows.reduce((sum, r) => sum + (Number(r.totalHarga) || 0), 0);
    const totalDiskonRaw = rows.reduce((sum, r) => sum + (Number(r.diskon) || 0), 0);
    const totalDppRaw = rows.reduce((sum, r) => sum + (Number(r.dpp) || 0), 0);
    const totalDppLainRaw = rows.reduce((sum, r) => sum + (Number(r.dppLain) || 0), 0);
    const totalPpnRaw = rows.reduce((sum, r) => sum + (Number(r.ppn) || 0), 0);
    const totalPpnbmRaw = rows.reduce((sum, r) => sum + (Number(r.ppnbm) || 0), 0);
    const nilaiReturJmlBrgRaw = rows.reduce((sum, r) => sum + (Number(r.returJmlBrgJasa) || 0), 0);
    const nilaiReturRaw = rows.reduce((sum, r) => sum + (Number(r.returDpp) || 0), 0);
    const nilaiReturDiskonRaw = rows.reduce((sum, r) => sum + (Number(r.returDiskon) || 0), 0);
    const nilaiReturDppRaw = rows.reduce((sum, r) => sum + (Number(r.returDpp) || 0), 0);
    const nilaiReturDppLainRaw = rows.reduce((sum, r) => sum + (Number(r.returDppLain) || 0), 0);
    const nilaiReturPpnRaw = rows.reduce((sum, r) => sum + (Number(r.returPpn) || 0), 0);
    const nilaireturppnbmRaw = rows.reduce((sum, r) => sum + (Number(r.returPpnbm) || 0), 0);

    const totalJumlahBarang = Math.round(totalJumlahBarangRaw);
    const totalHarga = Math.round(totalHargaRaw);
    const totalDiskon = Math.round(totalDiskonRaw);
    const totalDpp = Math.round(totalDppRaw);
    const totalDppLain = Math.round(totalDppLainRaw);
    const totalPpn = Math.round(totalPpnRaw);
    const totalPpnbm = Math.round(totalPpnbmRaw);
    const nilaiReturJmlBrg = Math.round(nilaiReturJmlBrgRaw);
    const nilaiRetur = Math.round(nilaiReturRaw);
    const nilaiReturDiskon = Math.round(nilaiReturDiskonRaw);
    const nilaiReturDpp = Math.round(nilaiReturDppRaw);
    const nilaiReturDppLain = Math.round(nilaiReturDppLainRaw);
    const nilaiReturPpn = Math.round(nilaiReturPpnRaw);
    const nilaireturppnbm = Math.round(nilaireturppnbmRaw);

    setValue('objekFaktur', rows);
    setValue('totalJumlahBarang', totalJumlahBarang);
    setValue('totalHarga', totalHarga);
    setValue('totalDiskon', totalDiskon);
    setValue('totalDpp', totalDpp);
    setValue('totalDppLain', totalDppLain);
    setValue('totalPpn', totalPpn);
    setValue('totalPpnbm', totalPpnbm);

    setValue('nilaiReturJmlBrg', nilaiReturJmlBrg);
    setValue('nilaiRetur', nilaiRetur);
    setValue('nilaiReturDiskon', nilaiReturDiskon);
    setValue('nilaiReturDpp', nilaiReturDpp);
    setValue('nilaiReturDppLain', nilaiReturDppLain);
    setValue('nilaiReturPpn', nilaiReturPpn);
    setValue('nilaireturppnbm', nilaireturppnbm);
  }, [rows, setValue]);

  // Sync rows from rekamData.objekFaktur
  useEffect(() => {
    if (rekamData && Array.isArray(rekamData.objekFaktur) && rekamData.objekFaktur.length > 0) {
      const mapped = rekamData.objekFaktur.map((it: any, idx: number) => {
        // safe map & cast numeric fields
        const num = (v: any) => {
          if (v === null || v === undefined || v === '') return 0;
          const n = Number(String(v).replace(/,/g, ''));
          return Number.isFinite(n) ? n : 0;
        };
        return {
          id: String(it.id ?? `${Date.now()}-${idx}`),
          brgJasa: it.brgJasa ?? it.type ?? '',
          namaBrgJasa: it.namaBrgJasa ?? it.nama ?? '',
          kdBrgJasa: it.kdBrgJasa ?? it.kode ?? '',
          jmlBrgJasa: num(it.jmlBrgJasa),
          satuanBrgJasa: it.satuanBrgJasa ?? it.satuan ?? '',
          hargaSatuan: num(it.hargaSatuan),
          totalHarga: num(it.totalHarga),
          diskon: num(it.diskon),
          dpp: num(it.dpp),
          dppLain: num(it.dppLain),
          ppn: num(it.ppn),
          ppnbm: num(it.ppnbm),
          tarifPpn: num(it.tarifPpn),
          tarifPpnbm: num(it.tarifPpnbm),
          cekDppLain: it.cekDppLain ?? false,
          returJmlBrgJasa: num(it.returJmlBrgJasa),
          returDpp: num(it.returDpp),
          returDppLain: num(it.returDppLain),
          returPpn: num(it.returPpn),
          returPpnbm: num(it.returPpnbm),
        };
      });
      setRows(mapped);
      setValue('objekFaktur', mapped, { shouldDirty: false, shouldValidate: false });

      return;
    }

    if (!rekamData) {
      const existing = getValues('objekFaktur');
      if (!existing || existing.length === 0) {
        setRows([]);
      }
    }
  }, [rekamData, setValue, getValues]);

  console.log(rows);

  return (
    <>
      <Grid size={{ xs: 12 }} sx={{ mt: 3 }}>
        <Divider sx={{ fontWeight: 'bold', fontSize: '1rem', mb: 2 }} textAlign="left">
          Detail Transaksi
        </Divider>
      </Grid>

      <DataGridPremium
        columns={columns}
        apiRef={apiRef}
        rows={rows}
        initialState={{
          pinnedColumns: {
            left: ['actions'],
          },
        }}
        slots={{
          // eslint-disable-next-line react/jsx-no-useless-fragment
          toolbar: () => <></>, // hilangkan toolbar filter/export
          noRowsOverlay: CustomNoRowsOverlay,
        }}
        sx={{
          border: 1,
          borderColor: 'divider',
          borderRadius: 2,
          mt: 3,
          minHeight: 'auto',
          '& .MuiDataGrid-cell': {
            borderColor: 'divider',
            userSelect: 'text',
            cursor: 'text',
          },
          '& .MuiDataGrid-columnHeaders': { borderColor: 'divider' },
        }}
        disableVirtualization
      />

      <ModalDetailTransaksiBarang
        key={`${editRow ? 'edit' : 'add'}-${goodsQuery.data?.length}-${servicesQuery.data?.length}-${satuanQuery.data?.length}`}
        isOpen={openModal}
        onClose={() => setOpenModal(false)}
        payload={{
          goods: goodsQuery.data,
          services: servicesQuery.data,
          satuan: satuanQuery.data,
        }}
        detailTransaksi={detailTransaksi}
        onSave={handleSaveFromModal}
        defaultValues={editRow ?? undefined}
      />
    </>
  );
};

export default DetailTransaksi;
