import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
import { paths } from 'src/routes/paths';
import React, { useCallback, useEffect, useMemo, useRef, useState, startTransition } from 'react';
import type {
  GridColDef,
  GridFilterModel,
  GridRowSelectionModel,
  GridSortModel,
  GridToolbarProps,
  GridPaginationModel,
} from '@mui/x-data-grid-premium';
import { DataGridPremium, useGridApiRef } from '@mui/x-data-grid-premium';
import { unstable_batchedUpdates } from 'react-dom';

import {
  ArticleTwoTone,
  AutorenewTwoTone,
  HighlightOffTwoTone,
  SearchRounded,
} from '@mui/icons-material';

import { Controller, FormProvider, useForm } from 'react-hook-form';
import { DateRangePicker, LocalizationProvider } from '@mui/x-date-pickers-pro';
import { AdapterDayjs } from '@mui/x-date-pickers-pro/AdapterDayjs';
import { enqueueSnackbar } from 'notistack';
import { usePaginationStore } from '../store/paginationStore';
import StatusChip from '../components/StatusChip';
import { useDebounce, useThrottle } from 'src/shared/hooks/useDebounceThrottle';
import useAdvancedFilter from '../hooks/useAdvancedFilterFakturReturPM';
import { CustomToolbar } from '../components/CustomToolbar';
import useGetReturFakturPK, { formatDateToDDMMYYYY } from '../hooks/useGetReturFakturPK';
import Button from '@mui/material/Button';
import ModalPrepolulatedReturPK from '../components/dialog/ModalPrepopulatedReturPK';
import ModalCancelReturPK from '../components/dialog/ModalCancelReturPK';
import ModalCetakReturPK from '../components/dialog/ModalCetakReturPK';
import Box from '@mui/material/Box';
import { Field } from 'src/components/hook-form';
import MenuItem from '@mui/material/MenuItem';

import { LoadingButton } from '@mui/lab';
import type { Dayjs } from 'dayjs';
import dayjs from 'dayjs';
import { formatRupiah } from 'src/shared/FormatRupiah/FormatRupiah';

export type IColumnGrid = GridColDef & {
  field:
    | 'statusretur'
    | 'nomorretur'
    | 'nomorfakturdiretur'
    | 'masapajak'
    | 'tahunpajak'
    | 'tanggalfaktur'
    | 'npwppembeli'
    | 'tanggalretur'
    | 'masaretur'
    | 'tahunretur'
    | 'npwppembeli'
    | 'namapembeli'
    | 'nilaireturdpp'
    | 'nilaireturdpplain'
    | 'nilaireturppn'
    | 'nilaireturppnbm';

  valueOptions?: readonly (string | number | { value: string | number; label: string })[];
};

type FilterType =
  | 'perioderetur'
  | 'periodemasa'
  | 'nomorFaktur'
  | 'nomorRetur'
  | 'npwp'
  | 'customDate';

interface FormValues {
  filter: FilterType;

  periodeRetur: string | null;
  periode: string | null;

  kataKunci?: [Dayjs | null, Dayjs | null];

  nomorFaktur: string;
  nomorRetur: string;
  npwp: string;
  nitku: string;
}

export function ReturFakturPkListView() {
  const apiRef = useGridApiRef();

  const tableKey = 'retur-faktur-pk';

  const { tables, filters, setPagination, resetPagination, setFilter } =
    usePaginationStore.getState();

  const [filterModel, setFilterModel] = useState<GridFilterModel>({
    items: filters[tableKey]?.items ?? [],
  });

  const [sortModel, setSortModel] = useState<GridSortModel>([]);
  const [rowSelectionModel, setRowSelectionModel] = useState<GridRowSelectionModel | undefined>(
    undefined
  );

  const [isPrepopulatedOpen, setIsPrepopulatedOpen] = useState(false);

  const [isPreviewOpen, setIsPreviewOpen] = useState<boolean>(false);
  const [isCancelModalOpen, setIsCancelModalOpen] = useState<boolean>(false);
  const [previewPayload, setPreviewPayload] = useState<Record<string, any> | undefined>(undefined);

  const dataSelectedRef = useRef<any[]>([]);
  const [selectionVersion, setSelectionVersion] = useState(0);
  const [localPagination, setLocalPagination] = useState({
    page: tables[tableKey]?.page ?? 0,
    pageSize: tables[tableKey]?.pageSize ?? 10,
  });

  const [requestParams, setRequestParams] = useState<any>(null);

  const { buildAdvancedFilter, buildRequestParams } = useAdvancedFilter();

  const page = tables[tableKey]?.page ?? 0;
  const pageSize = tables[tableKey]?.pageSize ?? 10;

  const isSyncingRef = useRef(false);
  const isEqual = (a: any, b: any) => a === b || JSON.stringify(a) === JSON.stringify(b);

  const methods = useForm<FormValues>({
    defaultValues: {
      filter: 'perioderetur',
      periodeRetur: null,
      periode: null,
      nomorFaktur: '',
      nomorRetur: '',
      npwp: '',
      nitku: '',
    },
  });

  const { watch } = methods;
  const selectedFilter = watch('filter');

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const newStoreItems = state.filters[tableKey]?.items ?? [];
      const localItems = filterModel.items ?? [];

      if (!isEqual(newStoreItems, localItems)) {
        isSyncingRef.current = true;
        setFilterModel({ items: newStoreItems });
        queueMicrotask(() => (isSyncingRef.current = false));
      }
    });

    return () => unsub();
  }, [filterModel.items]);

  useEffect(() => {
    const unsub = usePaginationStore.subscribe((state) => {
      const storePage = state.tables[tableKey]?.page ?? 0;
      const storePageSize = state.tables[tableKey]?.pageSize ?? 10;

      setLocalPagination((prev) =>
        prev.page !== storePage || prev.pageSize !== storePageSize
          ? { page: storePage, pageSize: storePageSize }
          : prev
      );
    });
    return () => unsub();
  }, []);

  useEffect(() => {
    methods.resetField('periodeRetur');
    methods.resetField('periode');
    methods.resetField('nomorFaktur');
    methods.resetField('nomorRetur');
    methods.resetField('npwp');
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [selectedFilter]);

  useEffect(() => {
    if (isSyncingRef.current) return;
    const currentStore = usePaginationStore.getState().filters[tableKey]?.items ?? [];
    if (!isEqual(currentStore, filterModel.items)) {
      setFilter(tableKey, { items: filterModel.items });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [filterModel]);

  const params = useMemo(() => {
    const advanced = buildAdvancedFilter(filterModel.items);

    return buildRequestParams(
      {
        page,
        limit: pageSize,
        sortingMode: sortModel[0]?.field ?? '',
        sortingMethod: sortModel[0]?.sort ?? '',
        ...requestParams,
      },
      advanced
    );
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [
    page,
    pageSize,
    sortModel,
    filterModel.items,
    requestParams, // ⬅️ WAJIB
  ]);

  const { data, isFetching, refetch } = useGetReturFakturPK({
    params,
  });
  const idStatusMapRef = useRef<Map<string | number, string>>(new Map());
  const rows = useMemo(() => (data?.data || []).filter(Boolean), [data?.data]);
  const totalRows = Number(data?.total ?? 0);

  useEffect(() => {
    try {
      const m = new Map<string | number, string>();
      (rows || []).forEach((r: any) => {
        const id = String(r.id ?? r.internal_id ?? '');
        m.set(id, r?.fgStatus ?? '');
      });
      idStatusMapRef.current = m;
    } catch {
      idStatusMapRef.current = new Map();
    }
  }, [rows]);

  const handlePaginationChange = (model: GridPaginationModel) => {
    setLocalPagination(model);
    setPagination(tableKey, {
      page: model.page,
      pageSize: model.pageSize,
    });
  };

  const debouncedFilterChange = useDebounce((model: GridFilterModel) => {
    setFilterModel(model);
  }, 400);

  const debouncedSortChange = useDebounce((model: GridSortModel) => {
    setSortModel(model);
    resetPagination(tableKey);
  }, 400);

  type Status =
    | 'DRAFT'
    | 'APPROVED'
    | 'WAITING FOR AMENDMENT'
    | 'AMENDED'
    | 'WAITING FOR CANCELLATION';
  type StatusOption = { value: Status; label: string };
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const statusOptions: StatusOption[] = [
    { value: 'DRAFT', label: 'Draft' },
    { value: 'APPROVED', label: 'Normal' },
    { value: 'WAITING FOR AMENDMENT', label: 'Normal Pengganti' },
    { value: 'AMENDED', label: 'Diganti' },
    { value: 'WAITING FOR CANCELLATION', label: 'Batal' },
  ];

  const columns = useMemo<IColumnGrid[]>(
    () => [
      {
        field: 'statusretur',
        headerName: 'Status',
        width: 260,
        type: 'singleSelect',
        valueOptions: statusOptions.map((opt) => opt.value),
        valueFormatter: ({ value }: { value: string }) => {
          const option = statusOptions.find((opt) => opt.value === value);
          return option ? option.label : value;
        },
        renderCell: ({ value, row }) => (
          <StatusChip value={value} credit={row.statuspembeli} valid={row.buyerstatus} />
        ),
      },
      { field: 'nomorretur', headerName: 'Nomor Retur Pajak', width: 200 },
      {
        field: 'nomorfakturdiretur',
        headerName: 'Nomor Faktur Pajak',
        width: 200,
      },

      {
        field: 'masapajak',
        headerName: 'Masa Pajak',
        width: 150,
      },
      {
        field: 'tahunpajak',
        headerName: 'Tahun Pajak',
        width: 150,
      },
      {
        field: 'tanggalfaktur',
        headerName: 'Tanggal Faktur Pajak',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.tanggalfaktur),
      },
      {
        field: 'masaretur',
        headerName: 'Masa Retur',
        width: 150,
      },
      {
        field: 'tahunretur',
        headerName: 'Tahun Retur',
        width: 150,
      },
      {
        field: 'tanggalretur',
        headerName: 'Tanggal Retur',
        width: 150,
        renderCell: ({ row }) => formatDateToDDMMYYYY(row.tanggalretur),
      },
      {
        field: 'npwppembeli',
        headerName: 'NPWP Pembeli',
        width: 150,
      },
      {
        field: 'namapembeli',
        headerName: 'Nama Pembeli',
        width: 150,
      },
      {
        field: 'nilaireturdpp',
        headerName: 'Nilai Retur DPP',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturdpp),
      },
      {
        field: 'nilaireturdpplain',
        headerName: 'Nilai Retur DPP Nilai Lain',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturdpplain),
      },
      {
        field: 'nilaireturppn',
        headerName: 'Nilai Retur PPN',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturppn),
      },
      {
        field: 'nilaireturppnbm',
        headerName: 'Nilai Retur Ppnbm',
        width: 150,
        renderCell: ({ row }) => formatRupiah(row.nilaireturppnbm),
      },
    ],
    [statusOptions]
  );

  const throttledSelectionChange = useThrottle((newSelection: any) => {
    if (!apiRef.current) return;
    const ids =
      newSelection?.ids instanceof Set ? Array.from(newSelection.ids) : newSelection || [];
    const selectedData = ids.map((id: any) => apiRef.current!.getRow(id)).filter(Boolean);

    unstable_batchedUpdates(() => {
      dataSelectedRef.current = selectedData;
      setRowSelectionModel(newSelection);
      setSelectionVersion((v) => v + 1);
    });
  }, 150);

  useEffect(() => {
    const api = apiRef.current;
    if (!api) return;
    const unsubscribe = api.subscribeEvent('rowsSet', () => {
      const exec = () => {
        const ids =
          api.state?.rowSelection?.ids instanceof Set ? Array.from(api.state.rowSelection.ids) : [];
        const updatedSelected = ids.map((id) => api.getRow(id)).filter(Boolean);
        dataSelectedRef.current = updatedSelected;
        setSelectionVersion((v) => v + 1);
      };
      if ((window as any).requestIdleCallback) (window as any).requestIdleCallback(exec);
      else setTimeout(exec, 0);
    });
    // eslint-disable-next-line consistent-return
    return () => unsubscribe();
  }, [apiRef]);

  const validatedActions = useMemo(() => {
    const dataSelected = dataSelectedRef.current;
    const count = dataSelected.length;

    if (count === 0) {
      return {
        canDetail: false,
        canCancel: false,
      };
    }

    const isSingle = count === 1;
    const allApproved = dataSelected.every(
      (d) => String(d.statusretur).toUpperCase() === 'APPROVED'
    );

    return {
      canDetail: isSingle,

      canCancel: allApproved,
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [selectionVersion]);

  const handleOpenPreview = () => {
    const selectedRow = dataSelectedRef.current?.[0];
    if (!selectedRow) {
      enqueueSnackbar('Pilih 1 baris untuk melihat detail', { variant: 'warning' });
      return;
    }

    console.log(selectedRow);

    setPreviewPayload(selectedRow);
    setIsPreviewOpen(true);
  };

  const actions = useMemo(
    () => [
      [
        {
          title: 'Refresh List',
          icon: <AutorenewTwoTone sx={{ width: 26, height: 26 }} />,
          func: () =>
            startTransition(() => {
              resetPagination(tableKey);
              void refetch();
            }),
        },
        {
          title: 'Detail',
          icon: <ArticleTwoTone sx={{ width: 26, height: 26 }} />,
          func: handleOpenPreview,
          disabled: !validatedActions.canDetail,
        },
      ],
      [
        {
          title: 'Batal',
          icon: <HighlightOffTwoTone sx={{ width: 26, height: 26 }} />,
          func: () => setIsCancelModalOpen(true),
          disabled: !validatedActions.canCancel,
        },
      ],
    ],
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [validatedActions, refetch]
  );

  const pinnedColumns = useMemo(
    () => ({ left: ['__check__', 'statusretur', 'nomorretur', 'nomorfakturdiretur'] }),
    []
  );

  const ToolbarWrapper: React.FC<GridToolbarProps> = useCallback(
    (gridToolbarProps) => (
      <CustomToolbar
        actions={actions}
        columns={columns}
        filterModel={filterModel}
        setFilterModel={setFilterModel}
        statusOptions={statusOptions}
        {...gridToolbarProps}
      />
    ),
    [actions, columns, filterModel, setFilterModel, statusOptions]
  );

  const onSubmit = (values: FormValues) => {
    const nextParams: any = {};

    if (values.filter === 'perioderetur' && values.periodeRetur) {
      const d = dayjs(values.periodeRetur);
      nextParams.perioderetur = JSON.stringify(d.toISOString());
      nextParams.periodeRetur = d.format('MM/YYYY');
    }

    if (values.filter === 'periodemasa' && values.periode) {
      const d = dayjs(values.periode);
      nextParams.periodemasa = JSON.stringify(d.toISOString());
      nextParams.periode = d.format('MM/YYYY');
    }

    if (values.filter === 'nomorFaktur' && values.nomorFaktur) {
      nextParams.nomorFaktur = values.nomorFaktur.trim();
    }

    if (values.filter === 'nomorRetur' && values.nomorRetur) {
      nextParams.nomorRetur = values.nomorRetur.trim();
    }

    if (values.filter === 'npwp' && values.npwp) {
      nextParams.npwp = values.npwp.replace(/\D/g, '');
    }

    if (values.filter === 'customDate' && values.kataKunci) {
      const [start, end] = values.kataKunci;

      if (start && end) {
        nextParams.startDate = start.startOf('day').toISOString();
        nextParams.endDate = end.endOf('day').toISOString();

        nextParams.tglAwal = start.format('YYYY-MM-DD');
        nextParams.tglAkhir = end.format('YYYY-MM-DD');
      }
    }

    // ⬅️ URUTAN PENTING
    resetPagination(tableKey); // reset ke page 0
    setRequestParams(nextParams); // trigger query
  };

  return (
    <>
      <DashboardContent>
        <CustomBreadcrumbs
          heading="Retur Pajak Keluaran"
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'Retur Pajak Keluaran' },
          ]}
          action={
            <Button variant="contained" onClick={() => setIsPrepopulatedOpen(true)}>
              Prepopulated
            </Button>
          }
        />
        <FormProvider {...methods}>
          <form onSubmit={methods.handleSubmit(onSubmit)}>
            <Box sx={{ mt: 4, display: 'flex', gap: 1, justifyContent: 'center' }}>
              <Box sx={{ width: 200 }}>
                <Field.Select name="filter" label="Pencarian Berdasarkan" size="medium">
                  <MenuItem value="perioderetur">Masa Retur Pajak</MenuItem>
                  <MenuItem value="periodemasa">Masa Faktur Pajak</MenuItem>
                  <MenuItem value="nomorFaktur">Nomor Faktur</MenuItem>
                  <MenuItem value="nomorRetur">Nomor Retur</MenuItem>
                  <MenuItem value="npwp">NPWP</MenuItem>
                  <MenuItem value="customDate">Custom (by Date)</MenuItem>
                </Field.Select>
              </Box>
              <Box sx={{ width: 300 }}>
                {selectedFilter === 'perioderetur' && (
                  <Field.DatePicker
                    name="periodeRetur"
                    label="Masa Retur Pajak"
                    views={['year', 'month']}
                    openTo="month"
                    format="MM/YYYY"
                  />
                )}

                {selectedFilter === 'periodemasa' && (
                  <Field.DatePicker
                    name="periode"
                    label="Masa Faktur Pajak"
                    views={['year', 'month']}
                    openTo="month"
                    format="MM/YYYY"
                  />
                )}

                {selectedFilter === 'nomorFaktur' && (
                  <Field.Text name="nomorFaktur" label="Nomor Faktur" />
                )}

                {selectedFilter === 'nomorRetur' && (
                  <Field.Text name="nomorRetur" label="Nomor Retur" />
                )}

                {selectedFilter === 'npwp' && (
                  <Field.Text
                    name="npwp"
                    label="NPWP"
                    // onChange={(e) => {
                    //   const value = e.target.value.replace(/\D/g, '').slice(0, 16);

                    // }}
                  />
                )}

                {selectedFilter === 'customDate' && (
                  <Controller
                    name="kataKunci"
                    control={methods.control}
                    defaultValue={[null, null]}
                    render={({ field, fieldState }) => (
                      <LocalizationProvider dateAdapter={AdapterDayjs}>
                        <DateRangePicker
                          value={field.value}
                          onChange={(newValue) => {
                            field.onChange(newValue);
                          }}
                          maxDate={dayjs()}
                          sx={{ width: 300 }}
                          calendars={1}
                          localeText={{ start: 'Start', end: 'End' }}
                          slotProps={{
                            textField: {
                              error: !!fieldState.error,
                              helperText: fieldState.error?.message,
                            },
                          }}
                        />
                      </LocalizationProvider>
                    )}
                  />
                )}
              </Box>
              <Box sx={{ width: 200 }}>
                <LoadingButton
                  size="xLarge"
                  type="submit"
                  variant="contained"
                  // loading={loading}
                  startIcon={<SearchRounded />}
                  loadingIndicator="Loading..."
                  // onClick={methods.handleSubmit(onSubmit)}
                >
                  Search
                </LoadingButton>
              </Box>
            </Box>
          </form>
        </FormProvider>

        <DataGridPremium
          apiRef={apiRef}
          checkboxSelection
          rows={rows || []}
          getRowId={(row: any) => {
            if (!row) return '';
            return String(row.id ?? row.internal_id ?? '');
          }}
          columns={columns}
          loading={isFetching}
          rowCount={totalRows}
          pagination
          paginationMode="server"
          paginationModel={localPagination}
          initialState={{
            pagination: {
              paginationModel: {
                page,
                pageSize,
              },
            },
            columns: {
              columnVisibilityModel: {
                jenisInvoice: true, // hide on load
              },
            },
          }}
          onPaginationModelChange={handlePaginationChange}
          pageSizeOptions={[5, 10, 15, 25, 50, 100]}
          filterMode="server"
          onFilterModelChange={debouncedFilterChange}
          sortingMode="server"
          onSortModelChange={debouncedSortChange}
          rowSelectionModel={rowSelectionModel}
          onRowSelectionModelChange={throttledSelectionChange}
          pinnedColumns={pinnedColumns}
          cellSelection
          showToolbar
          slots={{ toolbar: ToolbarWrapper }}
          sx={{
            border: 1,
            borderColor: 'divider',
            borderRadius: 2,
            mt: 3,
            '& .MuiDataGrid-cell': {
              borderColor: 'divider',
              userSelect: 'text',
              cursor: 'text',
            },
            '& .MuiDataGrid-columnHeaders': { borderColor: 'divider' },
          }}
          onColumnVisibilityModelChange={(newModel) => {
            if (newModel['__check__'] === false) {
              newModel['__check__'] = true;
            }
            apiRef.current?.setColumnVisibilityModel(newModel);
          }}
          disableVirtualization
        />
      </DashboardContent>

      {isPrepopulatedOpen && (
        <ModalPrepolulatedReturPK
          isOpenDialogUpload={isPrepopulatedOpen}
          setIsOpenDialogUpload={setIsPrepopulatedOpen}
          successMessage="Berhasil unduh retur PK prepopulated"
          onConfirmUpload={async () => {
            await refetch();
          }}
        />
      )}

      {isCancelModalOpen && (
        <ModalCancelReturPK
          dataSelected={rowSelectionModel}
          setSelectionModel={setRowSelectionModel}
          tableApiRef={apiRef}
          isOpenDialogCancel={isCancelModalOpen}
          setIsOpenDialogCancel={setIsCancelModalOpen}
          successMessage="Data berhasil dihapus"
          onConfirmCancel={async () => {
            await refetch();
          }}
        />
      )}

      {isPreviewOpen && (
        <ModalCetakReturPK
          payload={previewPayload}
          isOpen={isPreviewOpen}
          onClose={() => {
            setIsPreviewOpen(false);
            setPreviewPayload(undefined);
          }}
        />
      )}
    </>
  );
}
