import { LoadingButton } from '@mui/lab';
import {
  Dialog,
  DialogActions,
  DialogContent,
  DialogContentText,
  Stack,
  Alert,
} from '@mui/material';
import { useState } from 'react';
import Captcha from 'demos-react-captcha';

interface DialogConfirmProps {
  isOpen: boolean;
  handleClose: () => void;
  handleSubmit: () => void;
  title: string;
  description?: string;
  actionTitle?: string;
  isLoadingBtnSubmit?: boolean;
  isDisabledButtonSubmit?: boolean;
  children?: React.ReactNode;
  isCaptcha?: boolean;
  fullWidth?: boolean;
  maxWidth?: 'xs' | 'sm' | 'md' | 'lg' | 'xl';
}

export default function DialogConfirm({
  isOpen,
  handleClose,
  handleSubmit,
  title,
  description,
  actionTitle = 'Ya',
  isLoadingBtnSubmit = false,
  isDisabledButtonSubmit = false,
  children,
  isCaptcha = false,
  fullWidth = true,
  maxWidth = 'sm',
}: DialogConfirmProps) {
  const [isVerified, setIsVerified] = useState(false);
  const [alert, setAlert] = useState<{
    value: string;
    severity?: 'error' | 'success' | 'info' | 'warning';
  }>({ value: '' });

  // Validasi captcha sebelum submit
  const handleVerified = () => {
    if (!isVerified) {
      setAlert({
        value: 'Captcha yang dimasukkan tidak sesuai',
        severity: 'error',
      });
    } else {
      handleSubmit();
    }
  };

  return (
    <Dialog
      fullWidth={fullWidth}
      maxWidth={maxWidth}
      scroll="body"
      open={isOpen}
      onClose={handleClose}
      sx={{
        '& .MuiDialog-paper': {
          overflow: 'hidden',
          borderRadius: 2,
        },
      }}
    >
      <DialogContent sx={{ mt: 4 }}>
        {/* Alert jika captcha salah */}
        {alert.value && (
          <Stack sx={{ width: '100%', mb: 2 }} spacing={2}>
            <Alert
              variant="outlined"
              severity={alert.severity ?? 'error'} // default ke "error"
            >
              {alert.value}
            </Alert>
          </Stack>
        )}

        {/* Judul dialog */}
        <DialogContentText variant="h6" color="#000" align="center" sx={{ fontWeight: 600 }}>
          {title}
        </DialogContentText>

        {/* Deskripsi opsional */}
        {description && (
          <DialogContentText
            variant="body2"
            color="text.secondary"
            align="center"
            sx={{ mt: 1, mb: 2 }}
          >
            {description}
          </DialogContentText>
        )}

        {/* Captcha opsional */}
        {isCaptcha && (
          <div className="flex items-center justify-center mt-2 mb-3">
            <Captcha
              onRefresh={() => setIsVerified(false)}
              onChange={setIsVerified}
              placeholder="Masukkan captcha"
              length={5}
            />
          </div>
        )}

        {children}
      </DialogContent>

      {/* Tombol aksi */}
      <DialogActions sx={{ px: 3, pb: 2 }}>
        <Stack direction="row" justifyContent="end" spacing={2}>
          <LoadingButton
            onClick={handleClose}
            variant="text"
            sx={{ fontSize: '14px' }}
            size="medium"
          >
            Tidak
          </LoadingButton>
          <LoadingButton
            onClick={isCaptcha ? handleVerified : handleSubmit}
            variant="contained"
            color="primary"
            disabled={isDisabledButtonSubmit}
            loading={isLoadingBtnSubmit}
            size="medium"
            sx={{ fontSize: '14px' }}
          >
            {actionTitle}
          </LoadingButton>
        </Stack>
      </DialogActions>
    </Dialog>
  );
}
