import { zodResolver } from '@hookform/resolvers/zod';
import { LoadingButton } from '@mui/lab';
import Grid from '@mui/material/Grid';
import { Suspense, useState } from 'react';
import { FormProvider, useForm } from 'react-hook-form';
import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
import { paths } from 'src/routes/paths';
import HeadingRekam from 'src/shared/components/HeadingRekam';
import z from 'zod';
import Identitas from '../components/rekamDn/Identitas';
import Divider from '@mui/material/Divider';
import FormSkeleton from 'src/shared/skeletons/FormSkeleton';
import PphDipotong from '../components/rekamDn/PphDipotong';
import useGetKodeObjekPajak from '../hooks/useGetKodeObjekPajak';
import DokumenReferensi from '../components/rekamDn/DokumenReferensi';
import Agreement from 'src/shared/components/agreement/Agreement';
import Stack from '@mui/material/Stack';
import PanduanDnRekam from '../components/rekamDn/PanduanDnRekam';

const bpuSchema = z.object({
  tglPemotongan: z.string().nonempty('Tanggal Pemotongan wajib diisi'),
  thnPajak: z.string().nonempty('Tahun Pajak wajib diisi'),
  msPajak: z.string().nonempty('Masa Pajak wajib diisi'),
  nitku: z
    .string()
    .nonempty('NITKU wajib diisi')
    .regex(/^\d{22}$/, 'NITKU harus 22 digit'),
  namaDipotong: z.string().nonempty('Nama wajib diisi'),
  idDipotong: z
    .string()
    .nonempty('NPWP wajib diisi')
    .regex(/^\d{16}$/, 'NPWP harus 16 digit'),
  email: z.string().email({ message: 'Email tidak valid' }).optional(),
  noDokLainnya: z.string().nonempty('No Dokumen Lainnya wajib diisi'),
  // bisa tambah field lain sesuai kebutuhan
  keterangan1: z.string().optional(),
  keterangan2: z.string().optional(),
  keterangan3: z.string().optional(),
  keterangan4: z.string().optional(),
  keterangan5: z.string().optional(),
});

const DnRekamView = () => {
  const [isOpenPanduan, setIsOpenPanduan] = useState<boolean>(false);
  const [isCheckedAgreement, setIsCheckedAgreement] = useState<boolean>(false);

  const { data, isLoading, isError } = useGetKodeObjekPajak();

  type BpuFormData = z.infer<typeof bpuSchema>;

  const handleOpenPanduan = () => setIsOpenPanduan(!isOpenPanduan);

  const defaultValues = {
    tglPemotongan: '',
    thnPajak: '',
    msPajak: '',
    idDipotong: '',
    nitku: '',
    namaDipotong: '',
    email: '',
    keterangan1: '',
    keterangan2: '',
    keterangan3: '',
    keterangan4: '',
    keterangan5: '',
    kdObjPjk: '',
    fgFasilitas: '',
    noDokLainnya: '',
    jmlBruto: 0,
    tarif: '',
    pphDipotong: '',
    namaDok: '',
    nomorDok: '',
    tglDok: '',
    idTku: '',
  };

  const methods = useForm<BpuFormData>({
    mode: 'all',
    resolver: zodResolver(bpuSchema),
    defaultValues,
  });

  const {
    reset,
    handleSubmit,
    watch,
    formState: { isSubmitting },
  } = methods;

  //   const allValues = watch();

  //   console.log(allValues);
  const SubmitRekam = () => {
    console.log('Submit API');
  };

  return (
    <>
      <DashboardContent>
        <CustomBreadcrumbs
          heading="Add Bupot Unifikasi"
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'e-Bupot Unifikasi', href: paths.unifikasi.dn },
            { name: 'Add Bupot Unifikasi' },
          ]}
        />

        <HeadingRekam label="Rekam Data Bukti Pemotongan/Pemungutan PPh Unifikasi" />

        <Grid container columnSpacing={2} /* container otomatis */>
          <Grid size={{ xs: isOpenPanduan ? 8 : 11 }}>
            <form onSubmit={methods.handleSubmit(SubmitRekam)}>
              <FormProvider {...methods}>
                <Suspense fallback={<FormSkeleton />}>
                  <Identitas isPengganti={true} />
                  <Divider />
                  <Suspense fallback={<FormSkeleton />}>
                    <PphDipotong kodeObjectPajak={data?.data ?? []} />
                  </Suspense>
                  <DokumenReferensi />
                  <Divider />
                  <Grid size={12}>
                    <Agreement
                      isCheckedAgreement={isCheckedAgreement}
                      setIsCheckedAgreement={setIsCheckedAgreement}
                      text="Dengan ini saya menyatakan bahwa Bukti Pemotongan/Pemungutan Unifikasi telah
                      saya isi dengan benar secara elektronik sesuai
                      dengan"
                    />
                  </Grid>
                  <Stack direction="row" gap={2} justifyContent="end" marginTop={2}>
                    <LoadingButton
                      type="submit"
                      //   loading={saveDn.isLoading}
                      disabled={!isCheckedAgreement}
                      variant="outlined"
                      sx={{ color: '#143B88' }}
                    >
                      Save as Draft
                    </LoadingButton>
                    <LoadingButton
                      type="button"
                      disabled={!isCheckedAgreement}
                      //   onClick={handleClickUploadSsp}
                      //   loading={uploadDn.isLoading}
                      variant="contained"
                      sx={{ background: '#143B88' }}
                    >
                      Save and Upload
                    </LoadingButton>
                  </Stack>
                </Suspense>
              </FormProvider>
            </form>
          </Grid>
          <Grid size={{ xs: isOpenPanduan ? 4 : 1 }}>
            <PanduanDnRekam handleOpen={handleOpenPanduan} isOpen={isOpenPanduan} />
          </Grid>
        </Grid>
      </DashboardContent>
    </>
  );
};

export default DnRekamView;
