import { z } from 'zod';

export const DokumenLainKeluaranSchema = z
  .object({
    /* ================= HEADER ================= */
    kdTransaksi: z.string().min(1, 'Kode Transaksi wajib diisi'),
    detailTransaksi: z.string().min(1, 'Detail Transaksi wajib diisi'),

    fgPengganti: z.string().default('0'),

    refDoc: z.string().default(''),
    idKeteranganTambahan: z.string().nullable().default(null),

    /* ================= DOKUMEN ================= */
    dokumenTransaksi: z.string().min(1, 'Dokumen Transaksi wajib diisi'),
    kodeDokumen: z.string().min(1, 'Kode Dokumen wajib diisi'),
    nomorDokumen: z.string().min(1, 'Nomor Dokumen wajib diisi'),
    nomorDokumenDiganti: z.string().min(1, 'Nomor Dokumen Diganti wajib diisi'),
    keteranganTambahan: z.string().min(1, 'Keterangan Tambahan wajib diisi'),

    tanggalDokumen: z.coerce.date().refine((val) => !isNaN(val.getTime()), {
      message: 'Tanggal Dokumen wajib diisi',
    }),

    tanggalDokumenDiganti: z.coerce.date().nullable().optional(),

    masaPajak: z.string().min(1, 'Masa Pajak wajib diisi'),
    tahunPajak: z.string().min(1, 'Tahun Pajak wajib diisi'),

    referensi: z.string().min(1, 'Referensi wajib diisi'),

    /* ================= PEMBELI ================= */
    // idLainPembeli: z.string().default('0'),
    npwpPenjual: z.string().default(''),
    namaPenjual: z.string().min(1, 'Nama Penjual wajib diisi'),
    alamatPembeli: z.string().min(1, 'Alamat Pembeli wajib diisi'),
    // kdNegaraPembeli: z.string().min(1, 'Kode Negara wajib diisi'),
    // tkuPembeli: z.string().min(1, 'TKU Pembeli wajib diisi'),

    /* ================= TOTAL ================= */
    totalJumlahBarang: z.union([z.string(), z.number()]).transform(Number).default(0),
    totalHarga: z.union([z.string(), z.number()]).transform(Number).default(0),
    totalDiskon: z.union([z.string(), z.number()]).transform(Number).default(0),

    jumlahDpp: z.union([z.string(), z.number()]).transform(Number).default(0),
    // totalDppLain: z.union([z.string(), z.number()]).transform(Number).default(0),
    jumlahDppLain: z.union([z.string(), z.number()]).transform(Number).default(0),
    jumlahPpn: z.union([z.string(), z.number()]).transform(Number).default(0),
    jumlahPpnbm: z.union([z.string(), z.number()]).transform(Number).default(0),

    isPpnManual: z.boolean().default(false),

    /* ================= KETERANGAN ================= */
    keterangan1: z.string().default(''),
    keterangan2: z.string().default(''),
    keterangan3: z.string().default(''),
    keterangan4: z.string().default(''),
    keterangan5: z.string().default(''),

    /* ================= DETAIL ================= */
    objekFaktur: z.array(z.any()).default([]),

    approvalSign: z.string().min(1, 'NPWP/NIK Penandatangan wajib diisi'),
  })

  /* ================= CONDITIONAL VALIDATION ================= */
  .superRefine((data, ctx) => {
    const isExport = data.kdTransaksi === 'EXPORT';

    /* ===== NPWP RULE ===== */
    if (isExport) {
      if (data.npwpPenjual !== '0000000000000000') {
        ctx.addIssue({
          path: ['npwpPenjual'],
          code: 'custom',
          message: 'NPWP Pembeli untuk Ekspor harus 0000000000000000',
        });
      }
    } else {
      if (!/^\d{16}$/.test(data.npwpPenjual)) {
        ctx.addIssue({
          path: ['npwpPenjual'],
          code: 'custom',
          message: 'NPWP Pembeli harus 16 digit',
        });
      }
    }

    /* ===== EKSPOR BKP BERWUJUD ===== */
    if (isExport && data.detailTransaksi === 'TD.00311') {
      if (data.jumlahPpn !== 0 || data.jumlahPpnbm !== 0) {
        ctx.addIssue({
          path: ['jumlahPpn'],
          code: 'custom',
          message: 'PPN dan PPnBM untuk Ekspor BKP Berwujud harus 0',
        });
      }
    }
  });

export type TDokumenLainKeluaranSchema = z.infer<typeof DokumenLainKeluaranSchema>;
