import {
  useMutation,
  type UseMutationOptions,
  type UseMutationResult,
} from '@tanstack/react-query';
import type { AxiosError } from 'axios';
import dayjs from 'dayjs';
import { endpoints, fetcherCetakPDF } from 'src/lib/axios-ctas-box';
import { BUPOT_REFERENSI_TEXT, FG_FASILITAS_PPH_21_TEXT, KODE_OBJEK_PAJAK_TEXT } from './constant';
import queryKey from './constant/queryKey';

interface ApiCetakResponse {
  KdStatus: string;
  MsgStatus: string;
  arraybuff: string;
  url: string;
}

export interface transfromParamCetakBupotBulananProps {
  noBupot: string;
  masaPajak: string;
  tahunPajak: string;
  pasalPPh: string;
  status: string;
  statusPPh: string;
  npwpDipotong: string;
  namaDipotong: string;
  nitkuDipotong: string;
  sertifikatInsentifDipotong: string;
  kodeObjekPajak: string;
  namaObjekPajak: string;
  penghasilanBruto: string;
  tarif: string;
  pphDipotong: string;
  dokReferensi: string;
  nomorDokumen: string;
  tanggal_Dokumen: string;
  npwpPemotong: string;
  namaPemotong: string;
  nitkuPemotong: string;
  tglPemotongan: string;
  namaPenandatangan: string;
  nomorSertifikatInsentif: string;
  metodePembayaranBendahara: string;
  nomorSP2D: string;
  qrcode: string;
  mixcode: string;
}

export interface transFromPramsCetakBupotFinalTidakFinalProps {
  noBupot: string;
  masaPajak: string;
  tahunPajak: string;
  pasalPPh: string;
  status: string;
  statusPPh: string;
  npwp: string;
  namaDipotong: string;
  npwpDipotong: string;
  nitkuDipotong: string;
  sertifikatInsentifDipotong: string;
  kodeObjekPajak: string;
  namaObjekPajak: string;
  penghasilanBruto: string;
  tarif: string;
  pphDipotong: string;
  dokReferensi: string;
  nomorDokumen: string;
  tanggal_Dokumen: string;
  npwpPemotong: string;
  namaPemotong: string;
  nitkuPemotong: string;
  tglPemotongan: string;
  namaPenandatangan: string;
  nomorSertifikatInsentif: string;
  metodePembayaranBendahara: string;
  nomorSP2D: string;
  qrcode: string;
  mixcode: string;
}

export interface transFromCetakBupotTahuananA1 {
  qrcode: string;
  mixcode: string;
  noBupot: string;
  status: string;
  fgSatuPemberiKerja: string;
  fgStatusPemotonganPph: string;
  pasalPPh: string;
  masaPajakAwal: string;
  masaPajakAkhir: string;
  tahunPajak: string;
  npwpDipotong: string;
  namaDipotong: string;
  alamat: string;
  jnsKelamin: string;
  statusPtkp: string;
  jmlPtkp: string;
  nominalPtkp: string;
  kodeObjekPajak: string;
  pasalPph: string;
  fgKaryawanAsing: string;
  passport: string;
  kdNegara: string;
  posisiJabatan: string;
  gajiPensiun: string;
  tunjanganPPh: string;
  tunjanganLainnyaLembur: string;
  honorarium: string;
  premiAsuransi: string;
  natura: string;
  tantiemBonus: string;
  biayaJabatan: string;
  iuranPensiun: string;
  zakat: string;
  fgFasilitas: string;
  totalPenghasilanBruto: string;
  totalPengurang: string;
  totalPenghasilanNeto: string;
  totalPenghasilanNetoDariBupotSebelumnya: string;
  totalPenghasilanNetoPph21: string;
  pkpSetahunDisetahunkan: string;
  pph21SetahunDisetahunkan: string;
  pph21Terutang: string;
  pph21DariBupotSebelumnya: string;
  pph21DapatDikreditkan: string;
  pph21WithheldDtp: string;
  pph21KurangLebihBayar: string;
  tglPemotongan: string;
  namaPenandatangan: string;
  namaPemotong: string;
  npwpPemotong: string;
  nitkuPemotong: string;
}

type MutationProps<T> = Omit<
  UseMutationOptions<ApiCetakResponse, AxiosError, T>,
  'mutationKey' | 'mutationFn'
>;

const transformParamsBulanan = ({
  keterangan1,
  keterangan2,
  keterangan3,
  keterangan4,
  ...params
}: any): transfromParamCetakBupotBulananProps => ({
  ...params,
  noBupot: params.noBupot,
  masaPajak: params.msPajak,
  tahunPajak: params.thnPajak,
  pasalPPh: params.pasalPPh,
  status: 'Proforma',
  email: params.email || '',
  namaNegara: params.countryCode || '',
  statusPPh: params.statusPPh, // Final  atau Tidak Final
  npwpDipotong: params.npwp16Dipotong,
  namaDipotong: params.namaDipotong,
  nitkuDipotong: params.idDipotong,
  sertifikatInsentifDipotong: FG_FASILITAS_PPH_21_TEXT[params.fgFasilitas],
  nomorDokumen: params.noDokLainnya || '-',
  tanggal_Dokumen: '-',
  dokReferensi: '-',
  kodeObjekPajak: params.kdObjPjk,
  namaObjekPajak: KODE_OBJEK_PAJAK_TEXT[params.kdObjPjk],
  penghasilanBruto: `${Number(params.bruto) + Number(params.pph21ditanggungperusahaan)}`,
  tarif: `${params.tarif}`,
  pphDipotong: `${params.pphDipotong}`,
  npwpPemotong: params.npwp16Pemotong,
  namaPemotong: params.namaPemotong,
  nitkuPemotong: params.nitkuPemotong,
  tglPemotongan: dayjs(params.tglpemotongan).format('DD MMMM YYYY'),
  namaPenandatangan: params.namaPenandatangan,
  nomorSertifikatInsentif: params.noDokLainnya,
  metodePembayaranBendahara: '',
  nomorSP2D: '',
  qrcode: 'qrcode',
  mixcode: 'mixcode',
  link: '',
});

const kodeObjekPajakFinal = [
  '21-100-27',
  '21-100-29',
  '21-100-31',
  '21-100-37',
  '21-401-01',
  '21-401-02',
  '21-402-02',
  '21-402-03',
  '21-402-04',
];

const transformParamsFinalTidakFinal = ({
  keterangan1,
  keterangan2,
  keterangan3,
  keterangan4,
  ...params
}: any): transFromPramsCetakBupotFinalTidakFinalProps => ({
  ...params,
  noBupot: params.noBupot,
  pasalPPh: 'Final/Tidak Fianl',
  statusPPh: kodeObjekPajakFinal.includes(params.kdObjPjk) ? 'Final' : 'Tidak Final', // Final  atau Tidak Final
  status: 'Proforma',
  npwpDipotong: params.npwp,
  namaDipotong: params.nama,
  nitkuDipotong: params.idDipotong,
  sertifikatInsentifDipotong: FG_FASILITAS_PPH_21_TEXT[params.sertifikatInsentifDipotong],
  nomorDokumen: params.nomorDokumenReferensi || '-',
  tanggal_Dokumen: dayjs(params.tanggalDokumenReferensi).format('DD MM YYYY') || '-',
  dokReferensi: BUPOT_REFERENSI_TEXT[params.namaDokumenReferensi] || '-',
  namaObjekPajak: KODE_OBJEK_PAJAK_TEXT[params.kdObjPjk],
  penghasilanBruto: params.penghasilanKotor,
  tarif: `${params.tarif}`,
  pphDipotong: `${params.pphDipotong}`,
  npwpPemotong: params.npwpPemotong || '-',
  namaPemotong: params.namaPemotong || '-',
  nitkuPemotong: params.nitkuPemotong || '-',
  tglPemotongan: dayjs(params.tglpemotongan).format('DD MMMM YYYY'),
  namaPenandatangan: params.namaPenandatangan || '-',
  nomorSertifikatInsentif: params.noDokLainnya,
  metodePembayaranBendahara: 'metodePembayaranBendahara',
  nomorSP2D: '',
  qrcode: 'qrcode',
  mixcode: 'mixcode',
});

const { bulanan, finalTidakFinal, tahunanA1 } = endpoints.cetak;

const cetakBulanan = async (
  params: transfromParamCetakBupotBulananProps
): Promise<ApiCetakResponse> => {
  const response = await fetcherCetakPDF<ApiCetakResponse>([
    bulanan,
    {
      method: 'POST',
      data: transformParamsBulanan(params),
    },
  ]);

  if (!response.url) {
    throw new Error('Gagal cetak PDF Bulana ');
  }

  return response;
};

const cetakFinalTidakFinal = async (
  params: transFromPramsCetakBupotFinalTidakFinalProps
): Promise<ApiCetakResponse> => {
  const response = await fetcherCetakPDF<ApiCetakResponse>([
    finalTidakFinal,
    {
      method: 'POST',
      data: transformParamsFinalTidakFinal(params),
    },
  ]);

  if (!response.url) {
    throw new Error('Gagal cetak PDF Bulana ');
  }

  return response;
};

const cetakTahunanA1 = async (params: any): Promise<ApiCetakResponse> => {
  const response = await fetcherCetakPDF<ApiCetakResponse>([
    tahunanA1,
    {
      method: 'POST',
      data: params,
    },
  ]);

  if (!response.url) {
    throw new Error('Gagal cetak PDF Bulana ');
  }

  return response;
};

export function useCetakBulanan(
  props?: MutationProps<transfromParamCetakBupotBulananProps>
): UseMutationResult<ApiCetakResponse, AxiosError, transfromParamCetakBupotBulananProps> {
  return useMutation({
    mutationKey: queryKey.bulanan.cetakPdf(bulanan),
    mutationFn: cetakBulanan,
    ...props,
  });
}

export function useCetakFinalTidakFinal(
  props?: MutationProps<transFromPramsCetakBupotFinalTidakFinalProps>
): UseMutationResult<ApiCetakResponse, AxiosError, transFromPramsCetakBupotFinalTidakFinalProps> {
  return useMutation({
    mutationKey: queryKey.bulanan.cetakPdf(finalTidakFinal),
    mutationFn: cetakFinalTidakFinal,
    ...props,
  });
}

export function useCetakTahunanA1(
  props?: MutationProps<transFromCetakBupotTahuananA1>
): UseMutationResult<ApiCetakResponse, AxiosError, transFromCetakBupotTahuananA1> {
  return useMutation({
    mutationKey: queryKey.bulanan.cetakPdf(tahunanA1),
    mutationFn: cetakTahunanA1,
    ...props,
  });
}
