import { zodResolver } from '@hookform/resolvers/zod';
import { LoadingButton } from '@mui/lab';
import Grid from '@mui/material/Grid';
import React, { Suspense, useState } from 'react';
import { FormProvider, useForm } from 'react-hook-form';
import { CustomBreadcrumbs } from 'src/components/custom-breadcrumbs';
import { DashboardContent } from 'src/layouts/dashboard';
import { paths } from 'src/routes/paths';
import HeadingRekam from 'src/shared/components/HeadingRekam';
import z from 'zod';
import Divider from '@mui/material/Divider';
import FormSkeleton from 'src/shared/skeletons/FormSkeleton';
import useGetKodeObjekPajak from '../../bupot-dn/hooks/useGetKodeObjekPajak';
import DokumenReferensi from '../../bupot-dn/components/rekamDn/DokumenReferensi';
import Agreement from 'src/shared/components/agreement/Agreement';
import Stack from '@mui/material/Stack';
import PanduanNrRekam from '../components/rekamNr/PanduanNrRekam';
import Identitas from '../components/rekamNr/Identitas';
import PphDipotong from '../components/rekamNr/PphDipotong';

const bpuSchema = z.object({
  npwpPemotong: z.string().min(10, 'NPWP Pemotong wajib diisi'),
  idTku: z.string().min(5, 'ID TKU wajib diisi'),
  masaPajak: z.string().length(2, 'Masa Pajak harus 2 digit'),
  tahunPajak: z.string().length(4, 'Tahun Pajak harus 4 digit'),
  npwp: z.string().min(10, 'NPWP wajib diisi'),
  nik: z.string().min(10, 'NIK wajib diisi'),
  nama: z.string().min(2, 'Nama wajib diisi'),
  dpp: z.string().min(1, 'DPP wajib diisi'),
  tarif: z.string().min(1, 'Tarif wajib diisi'),
  pphDipotong: z.string().min(1, 'PPh Dipotong wajib diisi'),
  tglPemotongan: z.string().min(8, 'Format tgl: DDMMYYYY'),
  glAccount: z.string().min(3, 'GL Account wajib diisi'),
});

const NrRekamView = () => {
  const [isOpenPanduan, setIsOpenPanduan] = useState<boolean>(false);
  const [isCheckedAgreement, setIsCheckedAgreement] = useState<boolean>(false);

  const { data, isLoading, isError } = useGetKodeObjekPajak();

  type BpuFormData = z.infer<typeof bpuSchema>;

  const handleOpenPanduan = () => setIsOpenPanduan(!isOpenPanduan);

  const defaultValues = {
    npwpPemotong: '',
    idTku: '',
    masaPajak: '',
    tahunPajak: '',
    npwp: '',
    nik: '',
    nama: '',
    dpp: '',
    tarif: '',
    pphDipotong: '',
    tglPemotongan: '',
    glAccount: '',
    fgFasilitas: '',
  };

  const methods = useForm({
    mode: 'all',
    resolver: zodResolver(bpuSchema),
    defaultValues,
  });

  const {
    reset,
    handleSubmit,
    formState: { isSubmitting },
  } = methods;

  const SubmitRekam = () => {
    console.log('Submit API');
  };

  return (
    <>
      <DashboardContent>
        <CustomBreadcrumbs
          heading="Add Bupot Unifikasi Non Residen"
          links={[
            { name: 'Dashboard', href: paths.dashboard.root },
            { name: 'e-Bupot Unifikasi Non Residen', href: paths.unifikasi.nr },
            { name: 'Add Bupot Unifikasi Non Residen' },
          ]}
        />

        <HeadingRekam label="Rekam Data Bukti Potong PPh Non Residen" />

        <Grid container columnSpacing={2} /* container otomatis */>
          <Grid size={{ xs: isOpenPanduan ? 8 : 11 }}>
            <form onSubmit={methods.handleSubmit(SubmitRekam)}>
              <FormProvider {...methods}>
                <Suspense fallback={<FormSkeleton />}>
                  <Identitas isPengganti={true} />
                  <Divider />
                  <Suspense fallback={<FormSkeleton />}>
                    <PphDipotong kodeObjectPajak={data?.data ?? []} />
                  </Suspense>
                  <DokumenReferensi />
                  <Divider />
                  <Grid size={12}>
                    <Agreement
                      isCheckedAgreement={isCheckedAgreement}
                      setIsCheckedAgreement={setIsCheckedAgreement}
                      text="Dengan ini saya menyatakan bahwa Bukti Pemotongan/Pemungutan Unifikasi telah
                              saya isi dengan benar secara elektronik sesuai
                              dengan"
                    />
                  </Grid>
                  <Stack direction="row" gap={2} justifyContent="end" marginTop={2}>
                    <LoadingButton
                      type="submit"
                      //   loading={saveDn.isLoading}
                      disabled={!isCheckedAgreement}
                      variant="outlined"
                      sx={{ color: '#143B88' }}
                    >
                      Save as Draft
                    </LoadingButton>
                    <LoadingButton
                      type="button"
                      disabled={!isCheckedAgreement}
                      //   onClick={handleClickUploadSsp}
                      //   loading={uploadDn.isLoading}
                      variant="contained"
                      sx={{ background: '#143B88' }}
                    >
                      Save and Upload
                    </LoadingButton>
                  </Stack>
                </Suspense>
              </FormProvider>
            </form>
          </Grid>
          <Grid size={{ xs: isOpenPanduan ? 4 : 1 }}>
            <PanduanNrRekam handleOpen={handleOpenPanduan} isOpen={isOpenPanduan} />
          </Grid>
        </Grid>
      </DashboardContent>
    </>
  );
};

export default NrRekamView;
