import Grid from '@mui/material/Grid';
import dayjs from 'dayjs';
import { useEffect } from 'react';
import { useFormContext } from 'react-hook-form';
import { Field } from 'src/components/hook-form';
import type { TCountryResult } from '../../types/types';

type IdentitasProps = {
  isPengganti: boolean;
  existingNr?: any; // Data penuh dari API (opsional, untuk edit/pengganti)
  country: TCountryResult;
};

const Identitas = ({ isPengganti, existingNr, country }: IdentitasProps) => {
  const { setValue, watch } = useFormContext();

  const tanggalPemotongan = watch('tglPemotongan');

  // 🧩 Auto isi Tahun & Masa Pajak berdasarkan tanggalPemotongan
  useEffect(() => {
    if (tanggalPemotongan) {
      const date = dayjs(tanggalPemotongan);
      setValue('tglDok', tanggalPemotongan);
      setValue('thnPajak', date.format('YYYY'));
      setValue('masaPajak', date.format('MM'));
    } else {
      setValue('tglDok', '');
      setValue('thnPajak', '');
      setValue('masaPajak', '');
    }
  }, [tanggalPemotongan, setValue]);

  return (
    <>
      {/* 📋 Identitas Dasar */}
      <Grid container rowSpacing={2} alignItems="center" columnSpacing={2} sx={{ mb: 4 }}>
        {/* 📅 Tanggal & Masa Pajak */}
        <Grid size={{ md: 6 }}>
          <Field.DatePicker
            name="tglPemotongan"
            label="Tanggal Pemotongan"
            format="DD/MM/YYYY"
            maxDate={dayjs()}
            disabled={isPengganti}
          />
        </Grid>
        <Grid size={{ md: 3 }}>
          <Field.DatePicker
            name="thnPajak"
            label="Tahun Pajak"
            view="year"
            format="YYYY"
            disabled
          />
        </Grid>
        <Grid size={{ md: 3 }}>
          <Field.DatePicker
            name="masaPajak"
            label="Masa Pajak"
            views={['month']} // ✅ valid prop
            openTo="month"
            format="MM"
            disabled
          />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="idDipotong" label="Tax ID Number (TIN)" disabled={isPengganti} />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="namaDipotong" label="Nama" disabled={isPengganti} />
        </Grid>

        <Grid size={{ md: 12 }}>
          <Field.Text
            name="alamatDipotong"
            label="Alamat"
            multiline
            minRows={2}
            disabled={isPengganti}
            sx={{
              '& .MuiInputBase-inputMultiline': {
                lineHeight: 1.6,
              },
              '& .MuiOutlinedInput-root': {
                borderRadius: '8px',
              },
              '& .MuiOutlinedInput-notchedOutline': {
                borderColor: '#bdbdbd',
              },
              '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                borderColor: '#1976d2',
                borderWidth: '2px',
              },
            }}
          />
        </Grid>

        <Grid size={{ md: 12 }}>
          <Field.Autocomplete
            name="negaraDipotong"
            label="Negara"
            options={country}
          />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="tmptLahirDipotong" label="Tempat Lahir" disabled={isPengganti} />
        </Grid>
        <Grid size={{ md: 6 }}>
          <Field.DatePicker
            name="tglLahirDipotong"
            label="Tanggal Lahir"
            format="DD/MM/YYYY"
            maxDate={dayjs()}
            disabled={isPengganti}
          />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="nomorPaspor" label="No. Paspor" disabled={isPengganti} />
        </Grid>
        <Grid size={{ md: 6 }}>
          <Field.Text name="nomorKitasKitap" label="No.KITAS/KITAP" disabled={isPengganti} />
        </Grid>
      </Grid>
    </>
  );
};

export default Identitas;
