import Box from '@mui/material/Box';
import Button from '@mui/material/Button';
import Grid from '@mui/material/Grid';
import dayjs from 'dayjs';
import { useEffect, useState } from 'react';
import { useFormContext } from 'react-hook-form';
import { Field } from 'src/components/hook-form';
import type { TCountryResult } from '../../types/types';
import MenuItem from '@mui/material/MenuItem';

type IdentitasProps = {
  isPengganti: boolean;
  existingNr?: any; // Data penuh dari API (opsional, untuk edit/pengganti)
  country: TCountryResult;
};

const Identitas = ({ isPengganti, existingNr, country }: IdentitasProps) => {
  const { setValue, watch, getValues } = useFormContext();

  const tanggalPemotongan = watch('tglPemotongan');
  const maxKeterangan = 5;
  const [jumlahKeterangan, setJumlahKeterangan] = useState<number>(0);

  // 🧩 Auto isi Tahun & Masa Pajak berdasarkan tanggalPemotongan
  useEffect(() => {
    if (tanggalPemotongan) {
      const date = dayjs(tanggalPemotongan);
      setValue('thnPajak', date.format('YYYY'));
      setValue('masaPajak', date.format('MM'));
    } else {
      setValue('thnPajak', '');
      setValue('masaPajak', '');
    }
  }, [tanggalPemotongan, setValue]);

  // 🧠 Saat data API sudah masuk (edit/pengganti)
  // Gunakan getValues() agar langsung membaca nilai dari form (bukan nunggu watch)
  useEffect(() => {
    if (existingNr) {
      const currentValues = getValues();
      const arr = [
        currentValues.keterangan1,
        currentValues.keterangan2,
        currentValues.keterangan3,
        currentValues.keterangan4,
        currentValues.keterangan5,
      ];
      const count = arr.filter((k) => !!k && k.trim() !== '').length;

      console.log('🧠 Detected existing keterangan:', arr, 'count:', count);

      if (count > 0) {
        setJumlahKeterangan(count);
      }
    }
  }, [existingNr, getValues]);

  // 🧩 Pantau perubahan manual user (Tambah/Hapus)
  useEffect(() => {
    const subscription = watch((values) => {
      const arr = [
        values.keterangan1,
        values.keterangan2,
        values.keterangan3,
        values.keterangan4,
        values.keterangan5,
      ];
      const count = arr.filter((k) => !!k && k.trim() !== '').length;
      setJumlahKeterangan(count);
    });

    return () => subscription.unsubscribe();
  }, [watch]);

  // ➕ Tambah field baru
  const handleTambah = () => {
    if (jumlahKeterangan < maxKeterangan) {
      setJumlahKeterangan((prev) => prev + 1);
    }
  };

  // ➖ Hapus field terakhir
  const handleHapus = () => {
    if (jumlahKeterangan > 0) {
      setValue(`keterangan${jumlahKeterangan}`, '');
      setJumlahKeterangan((prev) => prev - 1);
    }
  };

  return (
    <>
      {/* 📋 Identitas Dasar */}
      <Grid container rowSpacing={2} alignItems="center" columnSpacing={2} sx={{ mb: 4 }}>
        {/* 📅 Tanggal & Masa Pajak */}
        <Grid size={{ md: 6 }}>
          <Field.DatePicker
            name="tglPemotongan"
            label="Tanggal Pemotongan"
            format="DD/MM/YYYY"
            maxDate={dayjs()}
            disabled={isPengganti}
          />
        </Grid>
        <Grid size={{ md: 3 }}>
          <Field.DatePicker
            name="thnPajak"
            label="Tahun Pajak"
            view="year"
            format="YYYY"
            disabled={isPengganti}
          />
        </Grid>
        <Grid size={{ md: 3 }}>
          <Field.DatePicker
            name="masaPajak"
            label="Masa Pajak"
            view="month"
            format="MM"
            disabled={isPengganti}
          />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="idDipotong" label="Tax ID Number (TIN)" disabled={isPengganti} />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="namaDipotong" label="Nama" disabled={isPengganti} />
        </Grid>

        <Grid size={{ md: 12 }}>
          <Field.Text
            name="alamatDipotong"
            label="Alamat"
            multiline
            minRows={2}
            disabled={isPengganti}
            sx={{
              '& .MuiInputBase-inputMultiline': {
                lineHeight: 1.6,
              },
              '& .MuiOutlinedInput-root': {
                borderRadius: '8px',
              },
              '& .MuiOutlinedInput-notchedOutline': {
                borderColor: '#bdbdbd',
              },
              '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
                borderColor: '#1976d2',
                borderWidth: '2px',
              },
            }}
          />
        </Grid>

        <Grid size={{ md: 12 }}>
          <Field.Select name="negaraDipotong" label="Negara" disabled={isPengganti}>
            {country.map((item) => (
              <MenuItem key={item.kode} value={item.kode}>
                {`${item.nama}`}
              </MenuItem>
            ))}
          </Field.Select>
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="tmptLahirDipotong" label="Tempat Lahir" disabled={isPengganti} />
        </Grid>
        <Grid size={{ md: 6 }}>
          <Field.DatePicker
            name="tglLahirDipotong"
            label="Tanggal Lahir"
            format="DD/MM/YYYY"
            maxDate={dayjs()}
            disabled={isPengganti}
          />
        </Grid>

        <Grid size={{ md: 6 }}>
          <Field.Text name="nomorPaspor" label="No. Paspor" disabled={isPengganti} />
        </Grid>
        <Grid size={{ md: 6 }}>
          <Field.Text name="nomorKitasKitap" label="No.KITAS/KITAP" disabled={isPengganti} />
        </Grid>
      </Grid>

      {/* ✏️ Tombol Tambah / Hapus Keterangan */}
      <Box sx={{ display: 'flex', gap: 2, mb: 3 }}>
        <Box
          sx={{
            borderRadius: '18px',
            border: jumlahKeterangan >= maxKeterangan ? '1px solid #eee' : '1px solid #2e7d3280',
            color: jumlahKeterangan >= maxKeterangan ? '#eee' : '#2e7d3280',
            p: '0px 10px',
          }}
        >
          <Button disabled={jumlahKeterangan >= maxKeterangan} onClick={handleTambah}>
            Tambah Keterangan
          </Button>
        </Box>
        <Box
          sx={{
            borderRadius: '18px',
            border: jumlahKeterangan === 0 ? '1px solid #eee' : '1px solid #f44336',
            color: jumlahKeterangan === 0 ? '#eee' : '#f44336',
            p: '0px 10px',
          }}
        >
          <Button disabled={jumlahKeterangan === 0} onClick={handleHapus}>
            Hapus Keterangan
          </Button>
        </Box>
      </Box>

      {/* 🗒️ Input Keterangan Tambahan */}
      <Box sx={{ mb: 3 }}>
        {Array.from({ length: jumlahKeterangan }).map((_, i) => (
          <Grid size={{ md: 12 }} key={`keterangan${i + 1}`}>
            <Field.Text
              sx={{ mb: 2 }}
              name={`keterangan${i + 1}`}
              label={`Keterangan Tambahan ${i + 1}`}
            />
          </Grid>
        ))}
      </Box>
    </>
  );
};

export default Identitas;
