import React, { useEffect, useState } from 'react';
import { useForm, FormProvider, useWatch, Controller } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';

import DialogUmum from 'src/shared/components/dialog/DialogUmum';
import DialogContent from '@mui/material/DialogContent';
import Box from '@mui/material/Box';
import Button from '@mui/material/Button';

import type {
  TGoods,
  TGoodsResult,
  TSatuanResult,
  TServiceResult,
  TServices,
} from '../../../types/types';

import InfoRoundedIcon from '@mui/icons-material/InfoRounded';
import type { TBarangJasaSchema } from '../../../schemas/detailBarangSchema';
import { BarangJasaSchema } from '../../../schemas/detailBarangSchema';
import Grid from '@mui/material/Grid';
import { RHFNumeric } from 'src/components/hook-form/rhf-numeric';
import FormLabel from '@mui/material/FormLabel';
import Stack from '@mui/material/Stack';
import RadioGroup from '@mui/material/RadioGroup';
import FormControlLabel from '@mui/material/FormControlLabel';
import Radio from '@mui/material/Radio';
import IconButton from '@mui/material/IconButton';
import Divider from '@mui/material/Divider';
import { Field } from 'src/components/hook-form';
import MenuItem from '@mui/material/MenuItem';
import AlertInformationDppLain from '../../AlertInformationDppLain';

interface ModalDetailTransaksiBarangProps {
  payload?: {
    goods?: TGoodsResult;
    services?: TServiceResult;
    satuan?: TSatuanResult;
  };
  isOpen: boolean;
  onClose: () => void;
  detailTransaksi?: string;
  onSave?: (obj: any) => void;
  defaultValues?: TBarangJasaSchema | null;
}

const DEFAULT_FORM: TBarangJasaSchema = {
  type: 'GOODS',
  kdBrgJasa: '',
  namaBrgJasa: '',
  satuanBrgJasa: '',
  jmlBrgJasa: '0',
  hargaSatuan: '0',
  totalHarga: '0',
  diskon: '0',
  fgPmk: '',
  dpp: '0',
  tarifPpn: '12',
  dppLain: '0',
  ppn: '0',
  tarifPpnbm: '0',
  ppnbm: '0',
};

const toString = (v: any, fallback = '0') => (v === undefined || v === null ? fallback : String(v));

const normalizeDefaultValues = (src?: any): TBarangJasaSchema => ({
  type: String(src?.type ?? src?.brgJasa ?? 'GOODS'),
  kdBrgJasa: src?.kdBrgJasa ?? src?.kode ?? '',
  namaBrgJasa: src?.namaBrgJasa ?? src?.nama ?? '',
  satuanBrgJasa: src?.satuanBrgJasa ?? src?.satuan ?? '',
  jmlBrgJasa: toString(src?.jmlBrgJasa ?? src?.jumlah ?? '0'),
  hargaSatuan: toString(src?.hargaSatuan ?? src?.harga ?? '0'),
  totalHarga: toString(src?.totalHarga ?? src?.total ?? '0'),
  diskon: toString(src?.diskon ?? '0'),
  fgPmk: src?.fgPmk ?? '',
  dpp: toString(src?.dpp ?? '0'),
  tarifPpn: toString(src?.tarifPpn ?? '12'),
  dppLain: toString(src?.dppLain ?? '0'),
  ppn: toString(src?.ppn ?? '0'),
  tarifPpnbm: toString(src?.tarifPpnbm ?? '0'),
  ppnbm: toString(src?.ppnbm ?? '0'),
});

const ModalDetailTransaksiBarang: React.FC<ModalDetailTransaksiBarangProps> = ({
  payload,
  isOpen,
  onClose,
  detailTransaksi = '',
  onSave,
  defaultValues = null,
}) => {
  const [alertInfo, setAlertInfo] = useState(true);

  const methods = useForm<TBarangJasaSchema>({
    mode: 'all',
    resolver: zodResolver(BarangJasaSchema),
    defaultValues: DEFAULT_FORM,
  });

  const { control, setValue, getValues, reset, handleSubmit } = methods;

  const type = useWatch({ control, name: 'type' });
  const kdBrgJasa = useWatch({ control, name: 'kdBrgJasa' });
  const jmlBrgJasa = useWatch({ control, name: 'jmlBrgJasa' });
  const hargaSatuan = useWatch({ control, name: 'hargaSatuan' });
  const diskon = useWatch({ control, name: 'diskon' });
  const fgPmk = useWatch({ control, name: 'fgPmk' });
  const tarifPpnbm = useWatch({ control, name: 'tarifPpnbm' });

  // LIST DATA
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const goods = payload?.goods ?? [];
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const services = payload?.services ?? [];
  // eslint-disable-next-line react-hooks/exhaustive-deps
  const satuan = payload?.satuan ?? [];

  const isPayloadReady =
    Boolean(payload) && (goods.length > 0 || services.length > 0) && satuan.length > 0;
  //-------------------------------------
  // ⛔ PERBAIKAN UTAMA:
  // Modal buka → tentukan ADD atau EDIT
  //-------------------------------------
  useEffect(() => {
    if (!isOpen) return;

    if (defaultValues) {
      reset(normalizeDefaultValues(defaultValues)); // EDIT
    } else {
      reset(DEFAULT_FORM); // ADD
    }
  }, [isOpen, defaultValues, reset]);

  //-------------------------------------
  // OPTIONS LIST
  //-------------------------------------
  type ItemKode = TGoods | TServices;
  const kodeList: ItemKode[] = React.useMemo(
    () => (type === 'GOODS' ? goods : services),
    [goods, services, type]
  );

  const filteredSatuan = React.useMemo(
    () =>
      satuan.filter((item: any) => {
        const num = Number(String(item.code).split('.')[1]);
        if (type === 'GOODS') return (num >= 1 && num <= 22) || (num >= 33 && num <= 39);
        if (type === 'SERVICES') return num >= 23 && num <= 33;
        return true;
      }),
    [satuan, type]
  );

  //-------------------------------------
  // AUTO FILL namaBrgJasa
  //-------------------------------------
  useEffect(() => {
    if (!kdBrgJasa) return;
    const selected = kodeList.find((it) => String(it.code) === String(kdBrgJasa));
    if (selected) {
      const nama = (selected as any).bahasa ?? '';
      setValue('namaBrgJasa', nama);
    }
  }, [kdBrgJasa, kodeList, setValue]);

  //-------------------------------------
  // AUTO TOTAL HARGA
  //-------------------------------------
  useEffect(() => {
    const jumlah = Number(jmlBrgJasa || 0);
    const harga = Number(hargaSatuan || 0);
    setValue('totalHarga', String(jumlah * harga));
  }, [jmlBrgJasa, hargaSatuan, setValue]);

  //-------------------------------------
  // AUTO DPP
  //-------------------------------------
  useEffect(() => {
    const jumlah = Number(jmlBrgJasa || 0);
    const harga = Number(hargaSatuan || 0);
    const disc = Number(diskon || 0);
    setValue('dpp', String(jumlah * harga - disc));
  }, [jmlBrgJasa, hargaSatuan, diskon, setValue]);

  //-------------------------------------
  // TARIF PPN FIX 12%
  //-------------------------------------
  useEffect(() => {
    setValue('tarifPpn', '12');
  }, [setValue]);

  //-------------------------------------
  // CLAMP TARIF PPnBM
  //-------------------------------------
  useEffect(() => {
    const raw = Number(tarifPpnbm || 0);
    const clamp = Math.max(0, Math.min(100, Math.round(raw)));
    setValue('tarifPpnbm', String(clamp));
  }, [tarifPpnbm, setValue]);

  //-------------------------------------
  // HITUNG DPP LAIN & PPN
  //-------------------------------------
  useEffect(() => {
    const dpp = Number(getValues('dpp') || 0);
    const tarifPpnbmNum = Number(getValues('tarifPpnbm') || 0);

    let dppLain = dpp;
    if (detailTransaksi !== 'TD.00301') {
      if (fgPmk === '0') dppLain = (dpp * 11) / 12;
      if (fgPmk === '1') dppLain = dpp;
    }

    const ppn = (dppLain * 12) / 100;
    const ppnbm = (dppLain * tarifPpnbmNum) / 100;

    setValue('dppLain', String(dppLain));
    setValue('ppn', String(ppn));
    setValue('ppnbm', String(ppnbm));
  }, [fgPmk, detailTransaksi, jmlBrgJasa, hargaSatuan, diskon, tarifPpnbm, getValues, setValue]);

  //-------------------------------------
  // SAVE
  //-------------------------------------
  const onSubmit = (data: TBarangJasaSchema) => {
    const entry = {
      ...data,
      brgJasa: data.type,
      jmlBrgJasa: Number(data.jmlBrgJasa),
      hargaSatuan: Number(data.hargaSatuan),
      totalHarga: Number(data.totalHarga),
      diskon: Number(data.diskon),
      dpp: Number(data.dpp),
      dppLain: Number(data.dppLain),
      tarifPpn: Number(data.tarifPpn),
      tarifPpnbm: Number(data.tarifPpnbm),
      ppn: Number(data.ppn),
      ppnbm: Number(data.ppnbm),
      cekDppLain: data.fgPmk === '0' || data.fgPmk === '1',
    };

    onSave?.(entry);

    // setelah save → kembali ke ADD
    reset(DEFAULT_FORM);
  };

  //-------------------------------------
  // CLOSE
  //-------------------------------------
  const handleClose = () => {
    reset(DEFAULT_FORM);
    onClose();
  };

  //-------------------------------------
  // RENDER
  //-------------------------------------
  return (
    <DialogUmum
      titleSx={{ borderBottom: '1px solid #eee', mb: 2 }}
      maxWidth="md"
      isOpen={isOpen}
      onClose={handleClose}
      title="Detail Penyerahan Barang/Jasa"
    >
      <FormProvider {...methods}>
        <DialogContent classes={{ root: 'p-16 sm:p-24' }}>
          <Grid container spacing={2} sx={{ mb: 3 }}>
            <Grid size={{ md: 12 }} sx={{ mt: 3 }}>
              <Divider sx={{ fontWeight: 'bold', fontSize: '1rem', mb: 2 }} textAlign="left">
                Detail Transaksi
              </Divider>
            </Grid>

            {/* Type Radio */}
            <Grid size={{ md: 4 }}>
              <Controller
                name="type"
                control={control}
                render={({ field }) => (
                  <>
                    <FormLabel sx={{ mb: 1, fontWeight: 600 }}>Type</FormLabel>
                    <RadioGroup row {...field}>
                      <FormControlLabel value="GOODS" control={<Radio />} label="Barang" />
                      <FormControlLabel value="SERVICES" control={<Radio />} label="Jasa" />
                    </RadioGroup>
                  </>
                )}
              />
            </Grid>

            <Grid size={{ md: 8 }} sx={{ display: 'flex', alignItems: 'end' }}>
              <Field.Select name="kdBrgJasa" label="Kode Barang/Jasa">
                {!isPayloadReady && goods.length === 0 && services.length === 0 ? (
                  <MenuItem disabled>Loading...</MenuItem>
                ) : kodeList.length > 0 ? (
                  kodeList.map((item, idx) => (
                    <MenuItem key={idx} value={(item as any).code}>
                      {`${(item as any).code} - ${(item as any).bahasa}`}
                    </MenuItem>
                  ))
                ) : (
                  <MenuItem disabled value="">
                    Tidak ada data
                  </MenuItem>
                )}
              </Field.Select>
            </Grid>

            <Grid size={{ md: 6 }}>
              <Field.Text name="namaBrgJasa" label="Nama Barang/Jasa" />
            </Grid>

            <Grid size={{ md: 6 }}>
              <Field.Select name="satuanBrgJasa" label="Satuan Barang/Jasa">
                {!isPayloadReady && !satuan.length ? (
                  <MenuItem disabled>Loading...</MenuItem>
                ) : filteredSatuan.length > 0 ? (
                  filteredSatuan.map((item, idx) => (
                    <MenuItem key={idx} value={(item as any).code}>
                      {`${(item as any).code} - ${(item as any).description}`}
                    </MenuItem>
                  ))
                ) : (
                  <MenuItem disabled value="">
                    Tidak ada data
                  </MenuItem>
                )}
              </Field.Select>
            </Grid>

            {/* Nilai Barang */}
            <Grid size={{ md: 6 }}>
              <RHFNumeric
                name="jmlBrgJasa"
                label="Jumlah Barang/Jasa"
                allowDecimalValue
                decimalScale={2}
              />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric
                name="hargaSatuan"
                label="Harga Satuan (Rp)"
                allowDecimalValue
                decimalScale={2}
              />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric
                name="totalHarga"
                label="Harga Total (Rp)"
                allowDecimalValue
                decimalScale={2}
              />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric name="diskon" label="Diskon (Rp)" allowDecimalValue decimalScale={2} />
            </Grid>

            {/* Divider */}
            <Grid size={{ md: 12 }} sx={{ mt: 3 }}>
              <Divider sx={{ fontWeight: 'bold', fontSize: '1rem', mb: 2 }} textAlign="left">
                PPN & PPnBM
              </Divider>
            </Grid>

            {alertInfo && (
              <Grid size={{ md: 12 }}>
                <AlertInformationDppLain onClose={() => setAlertInfo(false)} />
              </Grid>
            )}

            <Grid size={{ md: 6 }}>
              <RHFNumeric name="dpp" label="DPP (Rp)" allowDecimalValue decimalScale={2} readOnly />
            </Grid>

            <Grid size={{ md: 6 }}>
              <FormLabel sx={{ mb: 1, fontWeight: 600, display: 'block' }}>
                DPP Nilai Lain
              </FormLabel>
              <Stack direction="row" alignItems="center" spacing={1}>
                <Controller
                  name="fgPmk"
                  control={control}
                  render={({ field }) => (
                    <RadioGroup {...field} sx={{ flexDirection: 'column', gap: 0.5 }}>
                      <FormControlLabel
                        value="0"
                        control={<Radio />}
                        label="Sesuai Ketentuan PMK 131 Ps. 3 (2)"
                        disabled={detailTransaksi === 'TD.00301'}
                      />
                      <FormControlLabel
                        value="1"
                        control={<Radio />}
                        label="Selain yang diatur dalam PMK 131 Ps. 3 (2)"
                        disabled={detailTransaksi === 'TD.00301'}
                      />
                    </RadioGroup>
                  )}
                />
                <IconButton
                  color="primary"
                  onClick={() => setAlertInfo((p) => !p)}
                  sx={{ alignSelf: 'flex-start', mt: 0.5 }}
                >
                  <InfoRoundedIcon />
                </IconButton>
              </Stack>
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric name="tarifPpn" label="Tarif PPN %" allowDecimalValue={false} readOnly />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric
                name="dppLain"
                label="DPP Nilai Lain (Rp)"
                allowDecimalValue
                decimalScale={2}
                readOnly={detailTransaksi === 'TD.00301' || fgPmk !== '1'}
              />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric name="ppn" label="PPN (Rp)" allowDecimalValue decimalScale={2} />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric name="tarifPpnbm" label="Tarif PPnBM %" allowDecimalValue={false} />
            </Grid>

            <Grid size={{ md: 6 }}>
              <RHFNumeric name="ppnbm" label="PPnBM (Rp)" allowDecimalValue decimalScale={2} />
            </Grid>
          </Grid>

          {/* BAWAHNYA JUGA CUCI BERSIH TIDAK DIUBAH */}
          <Box sx={{ width: '100%', display: 'flex', justifyContent: 'right' }}>
            <Button
              type="button"
              onClick={handleSubmit(onSubmit)}
              variant="contained"
              sx={{ background: '#143B88' }}
            >
              Simpan
            </Button>
          </Box>
        </DialogContent>
      </FormProvider>
    </DialogUmum>
  );
};

export default ModalDetailTransaksiBarang;
